//===================================================================
//  
//  This driver is meant to be a simulation of a hprf scanner
//  device.  The driver itself is set up to make the programming 
//  API as simple as possible.
//  
//===================================================================

#include <windows.h>
#include <winbase.h>
#include <ceddk.h>
#include <winreg.h>

#include <initguid.h>
#include <types.h>
#include <ddkreg.h>
#include <serhw.h>

#include <initguid.h>

#include "ecspibus.h"
#include "bsp.h"

#include <eep_params.h>

#include "hprf_winceIntfc.h"
#include "smac_app.h"
#include "smac_mcu_hw_config.h"
#include "smac_mcu_spi_config.h"
#include "hprfgw_rfRec.h"
#include "hprfgw_rfTrans.h"
#include "hprfgw_rfApp.h"
#include "hprfgw_rfSlaveIntToHost.h"
#include "hprfgw_rfSlaveIntToRfApp.h"

#include "hprfgw_rfDiagnostics.h"

#include "UIHalAPI.h"
#include "nvram_dataMap.h"
#include "NvRamInterface.h"
//----------------------------------------------------------------------------
//
// public globals
//
//----------------------------------------------------------------------------

#ifdef DEBUG
    #undef ZONE_ERROR
    #undef ZONE_WARNING
    #undef ZONE_INIT
    #define ZONE_ERROR		DEBUGZONE(0)
    #define ZONE_WARNING	DEBUGZONE(1)
    #define ZONE_FUNCTION	DEBUGZONE(2)
    #define ZONE_INIT		DEBUGZONE(3)
    #define ZONE_DATA		DEBUGZONE(4)
    #define ZONE_IOCTL		DEBUGZONE(5)

    DBGPARAM dpCurSettings = { 
        TEXT("hprf"), 
        {
            TEXT("Error"),  TEXT("Warning"), TEXT("Function"), TEXT("Init"),
            TEXT("Data"),  TEXT("Ioctl"), TEXT("Undefined"), TEXT("Undefined"),
            TEXT("Undefined"),  TEXT("Undefined"), TEXT("Undefined"), TEXT("Undefined"),
            TEXT("Undefined"),  TEXT("Undefined"), TEXT("Undefined"), TEXT("Undefined") 
        },
        0x003f /* default - turn on all debug zones. */
    };

#endif /*DEBUG*/

//	#undef ZONE_ERROR
//	#undef ZONE_WARNING
//	#undef ZONE_INIT
//	#define ZONE_ERROR		TRUE
//	#define ZONE_WARNING	TRUE
//	#define ZONE_FUNCTION	TRUE
//	#define ZONE_INIT		TRUE
//	#define ZONE_DATA		TRUE
//	#define ZONE_IOCTL		TRUE

//----------------------------------------------------------------------------
//
// private globals
//
//----------------------------------------------------------------------------

// Allow a single instance of this driver to be loaded
static int DriverLoaded=0;

// Since this device is going to be designed such that only one program may
// read from a HPR device at a time, we will need to keep track of our
// open or closed state.  For this driver, if DevOpen!=0, we will assume 
// someone has opened the device for reading.
int OpenCount;

// These are the state variables that will hold the info taken from the 
// registry keys on how the driver should operate.  From the 4.1 lab, the
// keys should be set to:
// [HKEY_LOCAL_MACHINE/Software/ScanCo/DCD6000HPRFScanner]
//    "ScannerPowerMgt"=dword:0
//    "HPRFExtraTypes"=dword:1    
static int ScannerPowerMgt,HPRFExtraTypes;

// This will hold our power state.  Are we powered up, or not?
static int PoweredUp;

static CRITICAL_SECTION gCriticalSectionObject;

// This is a handle for the system memory pool for the entire RF to accomodate
// all the message queues, descriptive tables etc.
HANDLE hprSystemMemoryPool = NULL;

///////////////////////////////////////////////////
// Local functions
void  HPRF_LoadDStoreParams(void);
DWORD RFTx_CresnetPacketThread(void * pContext);
DWORD RFTx_LocalCresnetPacketThread(void * pContext);
DWORD RF_LQIThread(void * pContext);
INT32 RF_LQI_Timer_EchoRequest_CB(TYPE_RF_ECHO_REPLY_PACKET *pechoreply);
INT32 RF_LQI_Timer_NetworkService_CB(TYPE_RF_DISCOVER_REPLY *pRFDis, BOOL ignore);
BOOL  SetPowerState(hprf_hw_dev *pDevice, CEDEVICE_POWER_STATE dx);
DWORD RF_Diagnostics(void* pBufIn, DWORD dwLenIn, void* pBufOut, DWORD dwLenOut);
DWORD RF_Tests(void* pBufIn, DWORD dwLenIn, void* pBufOut, DWORD dwLenOut);
void  ReadProductIdAndRevision(DWORD* panelID, DWORD* revisionID);


// User EEPROM functions
BOOL HPRF_ReadWriteEEPROM(EEP_PARAMS* pEepParam, BOOL bReadWrite);
BOOL VerifyUserEepParams(void);

///////////////////////////////////////////////////
// SMAC debug timer register
#if SMAC_SM_DEBUG
 #ifdef SMAC_SM_DEBUG_TIMESTAMP_US
 // For SMAC_DEBUG purpose
 OMAP_GPTIMER_REGS *	pSmacDebugTimerReg;
 #endif
#endif

// Local structure to store registry values
RF_PARAM rfparam;
// Local structure only, no registry values
RF_PARAM_LOCAL rfparamlocal;

// Boot id message
CHAR bootidmsg[128];

// LQI global variable
UINT8   LocalDevLQI = 255, RemoteLQI = 255;
BOOL    g_bRF_LQI_Active = FALSE;

// Buffer to print ErrorPrintf messages & system errors
#ifdef HPRF_ERR_MSG_LOG
 TYPE_RF_ERR_LOG RfErrorLog;
#endif

// Extern variables
extern const unsigned char SMAC_Version[];
extern const unsigned char RF_Version[];

// Extern functions
extern void EnableECSPI();
extern void DisableECSPI();

//----------------------------------------------------------------------------
//
// DLL Entry
//
//----------------------------------------------------------------------------

BOOL 
DllEntry(
    HINSTANCE hinstDll,  // Process handle.
    DWORD   fdwReason,   // Reason for calling the function.
    LPVOID  lpvReserved  // Reserved, not used.
    )
{
    switch (fdwReason)
    {
        case DLL_PROCESS_ATTACH:
            DEBUGMSG(TRUE, (TEXT("HPRF.DLL: DLL_PROCESS_ATTACH\r\n")));
            DisableThreadLibraryCalls((HMODULE)hinstDll);
            DEBUGMSG(TRUE, (TEXT("HPRF.DLL: DLL_PROCESS_ATTACH\r\n")));
            break;

        case DLL_PROCESS_DETACH:
            DeleteCriticalSection(&gCriticalSectionObject);
            break;
    }

    return TRUE;
}


//----------------------------------------------------------------------------
//
// Standard Stream Device Driver Functions
//
//----------------------------------------------------------------------------
// The xxx_Init and xxx_Deinit functions are called at each instance load.  For
// example, if we had registry entries to create three bar code scanner devices
// named BAR1: BAR2: and BAR3:, we could expect HPR_Init to be called three 
// times.  This would require each instance of HPR_Init to create data structures
// specific to its particular incarnation, semaphores to lock shared regions, perform
// hardware initialization etc.  We limit this driver to a single instance.

hprf_hw_dev *pHPRDevice = NULL;

DWORD HPR_Init(DWORD dwContext)
{
    DWORD RetCode = FALSE;
    hprf_hw_dev *pDevice = NULL;
    MSGQUEUEOPTIONS mqOpt;
    DWORD panelID = 0, revisionID = 0;

#if SMAC_SM_DEBUG
    #ifdef SMAC_SM_DEBUG_TIMESTAMP_US
    // Map the OS timer register to a global variable and keep it ready for SMAC_DEBUG purpose
    PHYSICAL_ADDRESS pa;
    #endif
#endif
    DEBUGMSG(TRUE, (TEXT("HPRF.DLL: +HPR_Init\r\n")));

    // Check the revision ID and decide if we want to unload HPR DLL
    ReadProductIdAndRevision(&panelID, &revisionID);
    //RETAILMSG(TRUE, (TEXT("HPR_Init: Panel ID: 0x%x, Revision ID: 0x%x\r\n"), panelID, revisionID));
	switch(panelID)
	{
		case CRESTRON_TST600:
			switch(revisionID & TST600_HW_REV_WIFIRF_MASK)
			{
				case TST600_HW_REV_ID_TPMC_6X:
				case TST600_HW_REV_ID_TPMC_6XRF:
					break;
				default:
					DEBUGMSG(TRUE, (TEXT("ER device not supported by product ID (0x%x) and revision ID (0x%x).  Not initializing ER driver.\r\n"), panelID, revisionID));
					return FALSE;
			}
			break;
		case CRESTRON_TST900: //TODO: use revision bits to determine if 900 should load RF drivers
		case CRESTRON_TSR300:
			break;
		default:
			DEBUGMSG(TRUE, (TEXT("ER device not supported by product ID (0x%x) and revision ID (0x%x).  Not initializing ER driver.\r\n"), panelID, revisionID));
			return FALSE;
	}

	// Initialize global variables
    InitializeCriticalSection(&gCriticalSectionObject);
    //DriverLoaded = 0;

    EnterCriticalSection(&gCriticalSectionObject);
    if(DriverLoaded)
    {
        //Error: Driver is already loaded.
        LeaveCriticalSection(&gCriticalSectionObject);
        DEBUGMSG(ZONE_ERROR, (TEXT("HPRF.DLL: Already loaded, failing subsequent load!\r\n")));
        return FALSE;
    }

    DriverLoaded=1;
    OpenCount = 0;
    PoweredUp = 0;

    RetCode = TRUE;

    // Create a HPR system memory pool for all HPR stuff.
    //hprSystemMemoryPool = HeapCreate(0, HPR_INITIAL_HEAP_SIZE, HPR_MAX_HEAP_SIZE);
    //if (NULL == hprSystemMemoryPool)
    //{
    //	RFIntHostApp.pF_ErrorPrintf("HPR_Init: Could not allocate Heap.\n"); 
    //	return FALSE;
    //}

    // First create a local store for the hw params
    //pDevice = (hprf_hw_dev *)HeapAlloc(hprSystemMemoryPool, LPTR, sizeof(hprf_hw_dev));
    pDevice = (hprf_hw_dev *)LocalAlloc(LPTR, sizeof(hprf_hw_dev));
    if (NULL == pDevice)
    {
        RFIntHostApp.pF_ErrorPrintf("HPR_Init: Could not get heap for device.\n"); 
        return FALSE;
    }
    
    // Use a global pointer so that we can use it in all files.
    pHPRDevice = pDevice;   
    
    // This will be updated after the application starts, in the IOCTL
    pHPRDevice->RF_Rx_Queue_write = NULL;
    pHPRDevice->RF_Rx_Queue2_write = NULL;
    pHPRDevice->RF_Tx_Queue_read = NULL;
    pHPRDevice->RF_Tx_Queue2_read = NULL;
    pHPRDevice->RF_Cntrl_Queue_write = NULL;

    // Read the registry values and copy it to local structure.
    memset(&rfparam, 0, sizeof (rfparam));
	HPRF_LoadDStoreParams();
    // Set the default boot ID string
    strcpy(bootidmsg, RF_BOOT_ID_STRING_DEFAULT);

    /* Setup all the GPIOs
    PAEN   (input to radio)	: B9 (PATA_DATA0) 
    *RXEN  (input to radio)	: C5 (PATA_DATA3)
    *TX_EN (input to radio)	: V3 (PATA_DATA5)
    RXTXEN (input to radio)	: B8 (PATA_DATA1)
    *ATTN  (input to radio)	: B6 (PATA_DATA2)
    *RESET (input to radio)	: V2 (PATA_DATA6)
    *IRQ   (output from radio)	: V1 (PATA_DATA15)

    ID01   (FCC Id bit)		: W1 (PATA_DATA14)
    ID02   (FCC Id bit)		: AA1(PATA_DATA13)
    ID03   (FCC Id bit)		: Y1 (PATA_DATA12)

    PWR    (Radio Power ON)	: W2 (PATA_DATA7)    */

    //Do not enable the interrupt here. Will be done later.
    //DDKIomuxSetPinMux(DDK_IOMUX_PIN_PATA_DATA15, DDK_IOMUX_PIN_MUXMODE_ALT1, DDK_IOMUX_PIN_SION_REGULAR);
    //DDKGpioSetConfig(HPRF_GPIO_PORT, HPRF_GPIONUM_N_IRQ, DDK_GPIO_DIR_IN, DDK_GPIO_INTR_LOW_LEV);
    //DDKGpioClearIntrPin(HPRF_GPIO_PORT, HPRF_GPIONUM_N_IRQ);
    DDKIomuxSetPinMux(DDK_IOMUX_PIN_PATA_DATA15, DDK_IOMUX_PIN_MUXMODE_ALT1, DDK_IOMUX_PIN_SION_REGULAR);
	DDKIomuxSetPadConfig(DDK_IOMUX_PAD_PATA_DATA15, DDK_IOMUX_PAD_SLEW_FAST, DDK_IOMUX_PAD_DRIVE_MAX, 
						 DDK_IOMUX_PAD_OPENDRAIN_DISABLE, DDK_IOMUX_PAD_PULL_UP_100K, DDK_IOMUX_PAD_HYSTERESIS_ENABLE,
						 DDK_IOMUX_PAD_VDOEN_NULL, DDK_IOMUX_PAD_OUTVOLT_HIGH);
    
    DDKGpioSetConfig(HPRF_GPIO_PORT, HPRF_GPIONUM_N_IRQ, DDK_GPIO_DIR_IN, DDK_GPIO_INTR_NONE);
    
    DDKIomuxSetPinMux(DDK_IOMUX_PIN_PATA_DATA0, DDK_IOMUX_PIN_MUXMODE_ALT1, DDK_IOMUX_PIN_SION_REGULAR);
    DDKGpioSetConfig(HPRF_GPIO_PORT, HPRF_GPIONUM_PAEN, DDK_GPIO_DIR_OUT, DDK_GPIO_INTR_NONE);
    DDKGpioWriteDataPin(HPRF_GPIO_PORT, HPRF_GPIONUM_PAEN, 0);

    DDKIomuxSetPinMux(DDK_IOMUX_PIN_PATA_DATA3, DDK_IOMUX_PIN_MUXMODE_ALT1, DDK_IOMUX_PIN_SION_REGULAR);
    DDKGpioSetConfig(HPRF_GPIO_PORT, HPRF_GPIONUM_N_RXEN, DDK_GPIO_DIR_OUT, DDK_GPIO_INTR_NONE);
    DDKGpioWriteDataPin(HPRF_GPIO_PORT, HPRF_GPIONUM_N_RXEN, 0);

    DDKIomuxSetPinMux(DDK_IOMUX_PIN_PATA_DATA5, DDK_IOMUX_PIN_MUXMODE_ALT1, DDK_IOMUX_PIN_SION_REGULAR);
    DDKGpioSetConfig(HPRF_GPIO_PORT, HPRF_GPIONUM_N_TX_EN, DDK_GPIO_DIR_OUT, DDK_GPIO_INTR_NONE);
    DDKGpioWriteDataPin(HPRF_GPIO_PORT, HPRF_GPIONUM_N_TX_EN, 0);

    DDKIomuxSetPinMux(DDK_IOMUX_PIN_PATA_DATA1, DDK_IOMUX_PIN_MUXMODE_ALT1, DDK_IOMUX_PIN_SION_REGULAR);
    DDKGpioSetConfig(HPRF_GPIO_PORT, HPRF_GPIONUM_RXTXEN, DDK_GPIO_DIR_OUT, DDK_GPIO_INTR_NONE);
    DDKGpioWriteDataPin(HPRF_GPIO_PORT, HPRF_GPIONUM_RXTXEN, 0);
    
    DDKIomuxSetPinMux(DDK_IOMUX_PIN_PATA_DATA2, DDK_IOMUX_PIN_MUXMODE_ALT1, DDK_IOMUX_PIN_SION_REGULAR);
    DDKGpioSetConfig(HPRF_GPIO_PORT, HPRF_GPIONUM_N_ATTN, DDK_GPIO_DIR_OUT, DDK_GPIO_INTR_NONE);
    DDKGpioWriteDataPin(HPRF_GPIO_PORT, HPRF_GPIONUM_N_ATTN, 0);
    
    // Do not assert reset at this point.
    DDKIomuxSetPinMux(DDK_IOMUX_PIN_PATA_DATA6, DDK_IOMUX_PIN_MUXMODE_ALT1, DDK_IOMUX_PIN_SION_REGULAR);
    DDKGpioSetConfig(HPRF_GPIO_PORT, HPRF_GPIONUM_N_RESET, DDK_GPIO_DIR_OUT, DDK_GPIO_INTR_NONE);
    DDKGpioWriteDataPin(HPRF_GPIO_PORT, HPRF_GPIONUM_N_RESET, 1);

    // FCC ID pins
    DDKIomuxSetPinMux(DDK_IOMUX_PIN_PATA_DATA14, DDK_IOMUX_PIN_MUXMODE_ALT1, DDK_IOMUX_PIN_SION_REGULAR);
	DDKIomuxSetPadConfig(DDK_IOMUX_PAD_PATA_DATA14, DDK_IOMUX_PAD_SLEW_FAST, DDK_IOMUX_PAD_DRIVE_MAX, 
						 DDK_IOMUX_PAD_OPENDRAIN_DISABLE, DDK_IOMUX_PAD_PULL_UP_100K, DDK_IOMUX_PAD_HYSTERESIS_ENABLE,
						 DDK_IOMUX_PAD_VDOEN_NULL, DDK_IOMUX_PAD_OUTVOLT_HIGH);
    
    DDKGpioSetConfig(HPRF_GPIO_PORT, HPRF_GPIONUM_FCC_ID01, DDK_GPIO_DIR_IN, DDK_GPIO_INTR_NONE);
    
    DDKIomuxSetPinMux(DDK_IOMUX_PIN_PATA_DATA13, DDK_IOMUX_PIN_MUXMODE_ALT1, DDK_IOMUX_PIN_SION_REGULAR);
	DDKIomuxSetPadConfig(DDK_IOMUX_PAD_PATA_DATA13, DDK_IOMUX_PAD_SLEW_FAST, DDK_IOMUX_PAD_DRIVE_MAX, 
						 DDK_IOMUX_PAD_OPENDRAIN_DISABLE, DDK_IOMUX_PAD_PULL_UP_100K, DDK_IOMUX_PAD_HYSTERESIS_ENABLE,
						 DDK_IOMUX_PAD_VDOEN_NULL, DDK_IOMUX_PAD_OUTVOLT_HIGH);
    
    DDKGpioSetConfig(HPRF_GPIO_PORT, HPRF_GPIONUM_FCC_ID02, DDK_GPIO_DIR_IN, DDK_GPIO_INTR_NONE);

    DDKIomuxSetPinMux(DDK_IOMUX_PIN_PATA_DATA12, DDK_IOMUX_PIN_MUXMODE_ALT1, DDK_IOMUX_PIN_SION_REGULAR);
	DDKIomuxSetPadConfig(DDK_IOMUX_PAD_PATA_DATA12, DDK_IOMUX_PAD_SLEW_FAST, DDK_IOMUX_PAD_DRIVE_MAX, 
						 DDK_IOMUX_PAD_OPENDRAIN_DISABLE, DDK_IOMUX_PAD_PULL_UP_100K, DDK_IOMUX_PAD_HYSTERESIS_ENABLE,
						 DDK_IOMUX_PAD_VDOEN_NULL, DDK_IOMUX_PAD_OUTVOLT_HIGH);
    
    DDKGpioSetConfig(HPRF_GPIO_PORT, HPRF_GPIONUM_FCC_ID03, DDK_GPIO_DIR_IN, DDK_GPIO_INTR_NONE);

    // Radio power - RADIO_ON
    DDKIomuxSetPinMux(DDK_IOMUX_PIN_PATA_DATA7, DDK_IOMUX_PIN_MUXMODE_ALT1, DDK_IOMUX_PIN_SION_REGULAR);
    DDKIomuxSetPadConfig(DDK_IOMUX_PAD_PATA_DATA7, DDK_IOMUX_PAD_SLEW_SLOW, DDK_IOMUX_PAD_DRIVE_HIGH,
                         DDK_IOMUX_PAD_OPENDRAIN_DISABLE, DDK_IOMUX_PAD_PULL_NONE, DDK_IOMUX_PAD_HYSTERESIS_ENABLE,
                         DDK_IOMUX_PAD_VDOEN_NULL, DDK_IOMUX_PAD_OUTVOLT_NULL);
    DDKGpioSetConfig(HPRF_GPIO_PORT, HPRF_GPIONUM_RADIO_ON, DDK_GPIO_DIR_OUT, DDK_GPIO_INTR_NONE);
    DDKGpioWriteDataPin(HPRF_GPIO_PORT, HPRF_GPIONUM_RADIO_ON, 1);

    // Give some delay after swithing on the power
    RFIntHostApp.pF_HwDelayMsec(3);

    /* SD3_WP (EIM_D12) Debug GPIO pin - do not enable this when the SD card is present in
    the slot. This is at the SD card slot J9 on the PCB, which is the right most pin
    on the bottom, by holding the panel such that the microSD card slot is at upper right corner
    to the SD card slot and J9 is read in right way. */
    //DDKIomuxSetPinMux(DDK_IOMUX_PIN_EIM_DA12, DDK_IOMUX_PIN_MUXMODE_ALT1, DDK_IOMUX_PIN_SION_REGULAR);
    //DDKGpioSetConfig(DDK_GPIO_PORT3, 12, DDK_GPIO_DIR_OUT, DDK_GPIO_INTR_NONE);
    //DDKGpioWriteDataPin(DDK_GPIO_PORT3, 12, 0);

    // Lets open the handle to SPI bus here, to be safe. Configure later in the SPI init.
    // We are not using the driver came with the BSP. And All SPI initialization done in SMAC_SPIInit() in
    // panel specfici file
    /*pHPRDevice->hSPI = CSPIOpenHandle(HPR_SPI_BUS);
    if (NULL == pHPRDevice->hSPI) 
    {
        RFIntHostApp.pF_ErrorPrintf("HPR_Init(): Error getting SPI handle: %d\n", GetLastError());
    }*/
    // All other SPI configuration moved to SMAC_SPIInit()

#if SMAC_SM_DEBUG
    #ifdef SMAC_SM_DEBUG_TIMESTAMP_US
    // Map the OS timer register to a global variable and keep it ready for SMAC_DEBUG purpose
    pa.QuadPart = OMAP2430_GPTIMER1_REGS_PA;
    pSmacDebugTimerReg = (OMAP_GPTIMER_REGS *)MmMapIoSpace(pa, sizeof(OMAP_GPTIMER_REGS), FALSE);
    if (NULL == pSmacDebugTimerReg)
    {
        RFIntHostApp.pF_ErrorPrintf("HPR_Init: Error mapping SMAC_DEBUG timer");
    }
    #endif
#endif

    // update power state
    SetPowerState(pHPRDevice, D0);
    pHPRDevice->dwPowerDown = 0;

    // Init RF
    RFIntRFApp.pF_InitRFApplication();

    LeaveCriticalSection(&gCriticalSectionObject);

    ///////////////////////////////////////////////////////////////
    // Now create Application interface tasks and queues
    // RF Rx message queues
    // Note: If dwFlags set to NULL, then dwMaxMessages should be 0.
    // Or if dwMaxMessages set to non-zero, then dwFlags should be MSGQUEUE_ALLOW_BROKEN.
    mqOpt.dwSize = sizeof(mqOpt);
    mqOpt.dwFlags = MSGQUEUE_ALLOW_BROKEN | MSGQUEUE_NOPRECOMMIT;
    mqOpt.dwMaxMessages = 200;
    mqOpt.cbMaxMessage = 260;
    mqOpt.bReadAccess = FALSE;
    // Name should the same as that used in Application
    pHPRDevice->RF_Rx_Queue_write = CreateMsgQueue(_T("RFR_Q"),&mqOpt);
    if (pHPRDevice->RF_Rx_Queue_write == NULL)
    {
        RETAILMSG(TRUE, (TEXT("HPR_Init: Invalid Handle Rx Write Q: %d\r\n"), GetLastError()));
    }
    pHPRDevice->RF_Rx_Queue2_write = CreateMsgQueue(_T("RFR_Q2"),&mqOpt);
    if (pHPRDevice->RF_Rx_Queue2_write == NULL)
    {
        RETAILMSG(TRUE, (TEXT("HPR_Init: Invalid Handle Rx Write Q2: %d\r\n"), GetLastError()));
    }

    // RF Tx message queues
    // Note: If dwFlags set to NULL, then dwMaxMessages should be 0.
    // Or if dwMaxMessages set to non-zero, then dwFlags should be MSGQUEUE_ALLOW_BROKEN.
    mqOpt.dwSize = sizeof(mqOpt);
    mqOpt.dwFlags = MSGQUEUE_ALLOW_BROKEN | MSGQUEUE_NOPRECOMMIT;
    mqOpt.dwMaxMessages = 200;
    mqOpt.cbMaxMessage = 260;
    mqOpt.bReadAccess = TRUE;
    // Name should the same as that used in Application
    pHPRDevice->RF_Tx_Queue_read = CreateMsgQueue(_T("RFT_Q"),&mqOpt);
    if (pHPRDevice->RF_Tx_Queue_read == NULL)
    {
        RETAILMSG(TRUE, (TEXT("HPR_Init: Invalid Handle Tx Read Q: %d\r\n"), GetLastError()));
    }
    // Create another queue for Local cresnet packet
    pHPRDevice->RF_Tx_Queue2_read = CreateMsgQueue(_T("RFT_Q2"),&mqOpt);
    if (pHPRDevice->RF_Tx_Queue2_read == NULL)
    {
        RETAILMSG(TRUE, (TEXT("HPR_Init: Invalid Handle Tx Read Q2: %d\r\n"), GetLastError()));
    }

    //////////////////////////////////////////////////////////////
    ///////// We need to create a thread to wait on Transmit queue
    // Cresnet Packet Thread
    pHPRDevice->hRFTx_CresnetPacketThread = CreateThread(NULL, 0, RFTx_CresnetPacketThread, pHPRDevice, 0, NULL);
    if (!pHPRDevice->hRFTx_CresnetPacketThread)
    {
        RFIntHostApp.pF_ErrorPrintf("HPR_Init: Fail create CresnetPacketThread!");
    }
    else
    {
        // Set thread priority
        CeSetThreadPriority(pHPRDevice->hRFTx_CresnetPacketThread, RFTX_CRESNET_PACKET_TASK_PRIORITY);
    }

    // Local cresnet packet thread
    pHPRDevice->hRFTx_LocalCresnetPacketThread = CreateThread(NULL, 0, RFTx_LocalCresnetPacketThread, pHPRDevice, 0, NULL);
    if (!pHPRDevice->hRFTx_LocalCresnetPacketThread)
    {
        RFIntHostApp.pF_ErrorPrintf("HPR_Init: Fail create LocalCresnetPacketThread!");
    }
    else
    {
        // Set thread priority
        CeSetThreadPriority(pHPRDevice->hRFTx_LocalCresnetPacketThread, RFTX_LOCAL_CRESNET_PACKET_TASK_PRIORITY);
    }

    // RF Cntrl message queues
    // Note: If dwFlags set to NULL, then dwMaxMessages should be 0.
    // Or if dwMaxMessages set to non-zero, then dwFlags should be MSGQUEUE_ALLOW_BROKEN.
    mqOpt.dwSize = sizeof(mqOpt);
    mqOpt.dwFlags = MSGQUEUE_ALLOW_BROKEN | MSGQUEUE_NOPRECOMMIT;
    mqOpt.dwMaxMessages = 60;
    mqOpt.cbMaxMessage = 128;
    mqOpt.bReadAccess = FALSE;
    // Name should the same as that used in Application
    pHPRDevice->RF_Cntrl_Queue_write = CreateMsgQueue(_T("RFC_Q"),&mqOpt);
    if (pHPRDevice->RF_Cntrl_Queue_write == NULL)
    {
        RETAILMSG(TRUE, (TEXT("HPR_Init: Invalid Handle Ctrl Q: %d\r\n"), GetLastError()));
    }

    ///////////////////////////////////////////////////////////////////
    // RF Diag write queue (application reads it)
    // Note: If dwFlags set to NULL, then dwMaxMessages should be 0.
    // Or if dwMaxMessages set to non-zero, then dwFlags should be MSGQUEUE_ALLOW_BROKEN.
    mqOpt.dwSize = sizeof(mqOpt);
    mqOpt.dwFlags = MSGQUEUE_ALLOW_BROKEN | MSGQUEUE_NOPRECOMMIT;
    mqOpt.dwMaxMessages = 200;
    mqOpt.cbMaxMessage = 260;
    mqOpt.bReadAccess = FALSE;
    // Name should the same as that used in Application
    pHPRDevice->RF_Diag_Queue_write = CreateMsgQueue(_T("HPRFDIAG_Q"), &mqOpt);
    if (pHPRDevice->RF_Diag_Queue_write == NULL)
    {
        RETAILMSG(TRUE, (TEXT("HPR_Init: Invalid Handle HPRFDIAG_Q: %d\r\n"), GetLastError()));
    }

    ////////////////////////////////////////////////////
    //// LQI thread and events
    // Create manual reset event
    pHPRDevice->hLQIEvent = CreateEvent(NULL, TRUE, FALSE, NULL);
    if (pHPRDevice->hLQIEvent == NULL)
    {
        RETAILMSG(TRUE, (TEXT("HPR_Init: LQI Event Handle Fail!\r\n")));
    }
    // RF LQI thread
    pHPRDevice->hRF_LQIThread = CreateThread(NULL, RF_LQI_TASK_STACK_SIZE, RF_LQIThread,
                                                 pHPRDevice, STACK_SIZE_PARAM_IS_A_RESERVATION, NULL);
    if (!pHPRDevice->hRF_LQIThread)
    {
        RFIntHostApp.pF_ErrorPrintf("HPR_Init: Fail create RF LQI Thread!");
    }
    else
    {
        // Set thread priority
        CeSetThreadPriority(pHPRDevice->hRF_LQIThread, RF_LQI_TASK_PRIORITY);
    }
	
    ////////////////////////////////////////////////////
    DEBUGMSG(ZONE_FUNCTION, (TEXT("HPRF.DLL: -HPR_Init\r\n")));

    // Return nonzero device context if successful, 0 otherwise
    return (DWORD)pDevice; /* hDeviceContext */;
}

BOOL HPR_Deinit (DWORD hDeviceContext)
{
    hprf_hw_dev *pDevice = NULL;

    DEBUGMSG(ZONE_FUNCTION, (TEXT("HPRF.DLL: +HPR_Deinit\r\n")));
    DriverLoaded=0;
    pDevice = (hprf_hw_dev*)hDeviceContext;

    // TODO null reference checking
    // We are not using the driver came with the BSP.
    /*CloseHandle(pDevice->hSPI);
    pDevice->hSPI = INVALID_HANDLE_VALUE;*/

    LocalFree(pDevice);

    DEBUGMSG(ZONE_FUNCTION, (TEXT("HPRF.DLL: -HPR_Deinit\r\n")));
    return TRUE;
}		

// Allow multiple threads to open handles to this device.  There is nothing
// that we need to keep track of between open contexts, so just return an 
// incrementing number.
DWORD HPR_Open (DWORD hDeviceContext, DWORD AccessCode, DWORD ShareMode)
{
    //DEBUGMSG(ZONE_FUNCTION, (TEXT("HPRF.DLL: +HPR_Open\r\n")));
    //DEBUGMSG(ZONE_FUNCTION, (TEXT("HPRF.DLL: -HPR_Open\r\n")));

    // Return nonzero open context
    return ++OpenCount;
}

BOOL HPR_Close(DWORD hOpenContext)
{	
    //DEBUGMSG(ZONE_FUNCTION, (TEXT("HPRF.DLL: +HPR_Close\r\n")));
    //DEBUGMSG(ZONE_FUNCTION, (TEXT("HPRF.DLL: -HPR_Close\r\n")));
    // Decrement open handle to the driver
    --OpenCount;
    return TRUE;
}

void HPR_PowerDown (DWORD hDeviceContext)
{
    DEBUGMSG(ZONE_FUNCTION, (TEXT("HPRF.DLL: +HPR_PowerDown\r\n")));

	InterruptMask(pHPRDevice->dwHprfSysintr, 1);
	DDKGpioSetConfig(HPRF_GPIO_PORT, HPRF_GPIONUM_N_IRQ, DDK_GPIO_DIR_IN, DDK_GPIO_INTR_NONE);
    // Power Down the module
    RFIntRFApp.pF_PowerDown();	

    // Power down the hardware
    DDKGpioWriteDataPin(HPRF_GPIO_PORT, HPRF_GPIONUM_RADIO_ON, 0);

    // Also asserts the reset pin (otherwise, this injects some voltage in to the chip)
    DDKGpioWriteDataPin(HPRF_GPIO_PORT, HPRF_GPIONUM_N_RESET, 0);

	// As per HW engineers request, all input's into iMX53 should be set to a 360K internal 
	// pull down during power down mode.
	// Also, all the outputs should be driven low, namely
	// RF_TXEN, RF_RXEN, RF_RESET_N, RF_PA_EN, RF_RXTX_EN, RF_ATTN_N
	DDKIomuxSetPadConfig(DDK_IOMUX_PAD_PATA_DATA14, DDK_IOMUX_PAD_SLEW_FAST, DDK_IOMUX_PAD_DRIVE_MAX, 
						 DDK_IOMUX_PAD_OPENDRAIN_DISABLE, DDK_IOMUX_PAD_PULL_DOWN_360K, DDK_IOMUX_PAD_HYSTERESIS_ENABLE,
						 DDK_IOMUX_PAD_VDOEN_NULL, DDK_IOMUX_PAD_OUTVOLT_HIGH);
    
    DDKIomuxSetPadConfig(DDK_IOMUX_PAD_PATA_DATA13, DDK_IOMUX_PAD_SLEW_FAST, DDK_IOMUX_PAD_DRIVE_MAX, 
						 DDK_IOMUX_PAD_OPENDRAIN_DISABLE, DDK_IOMUX_PAD_PULL_DOWN_360K, DDK_IOMUX_PAD_HYSTERESIS_ENABLE,
						 DDK_IOMUX_PAD_VDOEN_NULL, DDK_IOMUX_PAD_OUTVOLT_HIGH);
    
    DDKIomuxSetPadConfig(DDK_IOMUX_PAD_PATA_DATA12, DDK_IOMUX_PAD_SLEW_FAST, DDK_IOMUX_PAD_DRIVE_MAX, 
						 DDK_IOMUX_PAD_OPENDRAIN_DISABLE, DDK_IOMUX_PAD_PULL_DOWN_360K, DDK_IOMUX_PAD_HYSTERESIS_ENABLE,
						 DDK_IOMUX_PAD_VDOEN_NULL, DDK_IOMUX_PAD_OUTVOLT_HIGH); 

	DDKIomuxSetPadConfig(DDK_IOMUX_PAD_PATA_DATA15, DDK_IOMUX_PAD_SLEW_FAST, DDK_IOMUX_PAD_DRIVE_MAX, 
						 DDK_IOMUX_PAD_OPENDRAIN_DISABLE, DDK_IOMUX_PAD_PULL_DOWN_360K, DDK_IOMUX_PAD_HYSTERESIS_ENABLE,
						 DDK_IOMUX_PAD_VDOEN_NULL, DDK_IOMUX_PAD_OUTVOLT_HIGH);

	// Drive all the outputs low
	DDKGpioWriteDataPin(HPRF_GPIO_PORT, HPRF_GPIONUM_PAEN, 0);
	DDKGpioWriteDataPin(HPRF_GPIO_PORT, HPRF_GPIONUM_N_RXEN, 0);
	DDKGpioWriteDataPin(HPRF_GPIO_PORT, HPRF_GPIONUM_N_TX_EN, 0);
	DDKGpioWriteDataPin(HPRF_GPIO_PORT, HPRF_GPIONUM_RXTXEN, 0);
	DDKGpioWriteDataPin(HPRF_GPIO_PORT, HPRF_GPIONUM_N_ATTN, 0);

    // Disable the ECSPI pheripheral
    DisableECSPI();

    DEBUGMSG(ZONE_FUNCTION, (TEXT("HPRF.DLL: -HPR_PowerDown\r\n")));
}

void HPR_PowerUp (DWORD hDeviceContext)
{
    DEBUGMSG(ZONE_FUNCTION, (TEXT("HPRF.DLL: +HPR_PowerUp\r\n")));

    // Enable the ECSPI pheripheral
    EnableECSPI();

	// FCC ID pins
	// Reinitialize the pins after power up
    DDKIomuxSetPadConfig(DDK_IOMUX_PAD_PATA_DATA14, DDK_IOMUX_PAD_SLEW_FAST, DDK_IOMUX_PAD_DRIVE_MAX, 
						 DDK_IOMUX_PAD_OPENDRAIN_DISABLE, DDK_IOMUX_PAD_PULL_UP_100K, DDK_IOMUX_PAD_HYSTERESIS_ENABLE,
						 DDK_IOMUX_PAD_VDOEN_NULL, DDK_IOMUX_PAD_OUTVOLT_HIGH);
    
    DDKIomuxSetPadConfig(DDK_IOMUX_PAD_PATA_DATA13, DDK_IOMUX_PAD_SLEW_FAST, DDK_IOMUX_PAD_DRIVE_MAX, 
						 DDK_IOMUX_PAD_OPENDRAIN_DISABLE, DDK_IOMUX_PAD_PULL_UP_100K, DDK_IOMUX_PAD_HYSTERESIS_ENABLE,
						 DDK_IOMUX_PAD_VDOEN_NULL, DDK_IOMUX_PAD_OUTVOLT_HIGH);
    
    DDKIomuxSetPadConfig(DDK_IOMUX_PAD_PATA_DATA12, DDK_IOMUX_PAD_SLEW_FAST, DDK_IOMUX_PAD_DRIVE_MAX, 
						 DDK_IOMUX_PAD_OPENDRAIN_DISABLE, DDK_IOMUX_PAD_PULL_UP_100K, DDK_IOMUX_PAD_HYSTERESIS_ENABLE,
						 DDK_IOMUX_PAD_VDOEN_NULL, DDK_IOMUX_PAD_OUTVOLT_HIGH); 

	DDKIomuxSetPadConfig(DDK_IOMUX_PAD_PATA_DATA15, DDK_IOMUX_PAD_SLEW_FAST, DDK_IOMUX_PAD_DRIVE_MAX, 
						 DDK_IOMUX_PAD_OPENDRAIN_DISABLE, DDK_IOMUX_PAD_PULL_UP_100K, DDK_IOMUX_PAD_HYSTERESIS_ENABLE,
						 DDK_IOMUX_PAD_VDOEN_NULL, DDK_IOMUX_PAD_OUTVOLT_HIGH);


    // First restore the state of reset pin (as it was before)
    DDKGpioWriteDataPin(HPRF_GPIO_PORT, HPRF_GPIONUM_N_RESET, 1);

    // Power up the hardware
    DDKGpioWriteDataPin(HPRF_GPIO_PORT, HPRF_GPIONUM_RADIO_ON, 1);

    RFIntRFApp.pF_PowerUp();
    
	DDKGpioSetConfig(HPRF_GPIO_PORT, HPRF_GPIONUM_N_IRQ, DDK_GPIO_DIR_IN, DDK_GPIO_INTR_LOW_LEV);
	InterruptMask(pHPRDevice->dwHprfSysintr, 0);	
	
	// Connect when  system power state changes (to resume).
	// During connect, HPR driver tries to open PDS driver and it causes system to hang 
	// on Resume when PDS driver is not up and running. 
	// Moved connect code to SetPowerState()

    DEBUGMSG(ZONE_FUNCTION, (TEXT("HPRF.DLL: -HPR_PowerUp\r\n")));
}

//------------------------------------------------------------------------------
//
//  Function:  SetPowerState
//
//  process the power state change request.
//
BOOL SetPowerState(hprf_hw_dev *pDevice, CEDEVICE_POWER_STATE dx)
{
	DEBUGMSG(ZONE_FUNCTION, (L"+HPRF: SetPowerState(0x%08x, Current: %d, To: %d)\r\n", pDevice, pDevice->powerState, dx));
        
    // only process request is the power states are different
    if (pDevice->powerState != dx)
    {
    	if( ((pDevice->powerState == D3) || (pDevice->powerState == D4)) &&
			((dx == D2) || (dx == D1) || (dx == D0)) )
    	{
	    	// Connect only if the UID is valid. Otherwise we get 
		    // "VMProcessPageFault Error: Page Fault occured  at power up" when no UID assigned
	    	if (rfparamlocal.UID_Valid == UID_SIGN && Smac_PhyInitialized)
		    {
	    	    //If we want to avoid sending connect which causes update request for unknown wakeup
	        	//For example, see \PLATFORM\MAINSTONEIII\SRC\DRIVERS\PWRBUTTON\pwrbutton.c
		        //KernelIoControl(IOCTL_HAL_GET_WAKE_SOURCE, NULL, 0, &wakeSrc, sizeof(wakeSrc), &bytesRet);

	    	    // Send connect message here itself. Other ways is to put it in SetPowerState when the
	        	// state changes from D3 to below. But with that it is noticed that the first key presses
		        // are sometimes missed sending to control system.
	    	    RFIntRFApp.pF_LookupSettingsAndConnectDeviceToNetwork();
    		}
    	}
 		
        pDevice->powerState = dx;

        // If it is D3 (suspend) set the flag to indicate not to process the
        // interrupts any more. This needs to be done here itself so that when the PowerDown function
        // gets called, the interrupt thread would have finished its current iteration (if running) and at
        // a known position. We get about 300 ticks from this point until the PowerDown function called.
        // If this is done in the PowerDown function, there is a possibility that the interrupt thread is in
        // the middle of something (PowerDown function called from PRIORITY 1 thread of windows power manager)
        // and during power up, it will start running from there and might read some garbage value which might
        // cause unexpcted things (we saw in conditions where control system is continously sending analog,
        // panel powers down and then during power up, sometimes we got undefined instruction exception and radio
        // module would stop working).
		// Commented out because this causes freezing issues when going into suspend.
#if 0
        if (pDevice->powerState == D3)
        {
            pDevice->dwPowerDown = 1;
        }
        else
        {
            // Clear the flag again here just for safety. This is already cleared in PowerUp function.
            pDevice->dwPowerDown = 0;
        }
#endif

    }
    return TRUE;
}

BOOL HPR_IOControl (
    DWORD hOpenContext,
    DWORD dwCode,
    PBYTE pBufIn,
    DWORD dwLenIn,
    PBYTE pBufOut,
    DWORD dwLenOut,
    PDWORD pdwActualOut )
{
    BOOL RetVal;
    DWORD dwReturn;
    DWORD dwData;

    // Variables needed for different cases
    UINT32 u32AcquireTimeout;
    DWORD dwTemp;
    UINT8  lq;
    UINT16 lq_norm;

    // Power management
    PPOWER_CAPABILITIES PowerCaps;
    CEDEVICE_POWER_STATE ReqDx;
    hprf_hw_dev *pDevice;

    // Create instance of EEPROM params
    EEP_PARAMS eep_param;
    
    ///////// End variable section /////////

    RetVal = FALSE;
    dwReturn = HPR_IOCTL_RESULT_SUCCESS;

    //pDevice = (hprf_hw_dev*)hOpenContext; // (this throws exception)
    pDevice = pHPRDevice;

    // TODO delete unwanted IOCTLs
    switch ( dwCode ) 
    {
        // In this sample driver, we will handle the power manually even though this is not typically
        // what you would do in a real-world driver.  In a real world driver you would implement
        // IOCTL_POWER_CAPABILITIES here, as well as IOCTL_POWER_GET and IOCTL_POWER_SET, which would
        // permit the power manager to handle the power for this device.  However, that would be beyond 
        // the scope of this simple driver, so for our purposes here we will handle the power to the
        // device manually, which will also serve as an example of how to handle IOCTL calls.  See the 
        // MSDN article on IOCTL_POWER_CAPABILITIES for the correct way to register with the Power Manager.

        case IOCTL_POWER_CAPABILITIES: 
            DEBUGMSG(ZONE_IOCTL, (L"HPR: IOCTL_POWER_CAPABILITIES\r\n"));
            if (pBufOut && dwLenOut >= sizeof (POWER_CAPABILITIES) && pdwActualOut)
            {
                __try 
                {
                    PowerCaps = (PPOWER_CAPABILITIES)pBufOut;
                    memset(PowerCaps, 0, sizeof(*PowerCaps));
                    PowerCaps->DeviceDx = (DX_MASK(D0) | DX_MASK(D1) | DX_MASK(D2) | DX_MASK(D3) | DX_MASK(D4));
                    *pdwActualOut = sizeof(*PowerCaps);                        
                    RetVal = TRUE;
                }
                __except(EXCEPTION_EXECUTE_HANDLER) 
                {
                    DEBUGMSG(ZONE_IOCTL, (L"HPR: Exception in ioctl\r\n"));
                }
            }
            break;

        // deprecated
        case IOCTL_POWER_QUERY:
            RetVal = TRUE;
            DEBUGMSG(ZONE_IOCTL,(L"HPR: IOCTL_POWER_QUERY\r\n"));
            break;

        // requests a change from one device power state to another
        case IOCTL_POWER_SET: 
            DEBUGMSG(ZONE_IOCTL,(L"HPR: IOCTL_POWER_SET\r\n"));
            if (pBufOut && dwLenOut >= sizeof(CEDEVICE_POWER_STATE)) 
            {
                __try 
                {
                    ReqDx = *(PCEDEVICE_POWER_STATE)pBufOut;
                    SetPowerState(pDevice, ReqDx);
                        
                    *pdwActualOut = sizeof(CEDEVICE_POWER_STATE);
                    RetVal = TRUE;
                }
                __except(EXCEPTION_EXECUTE_HANDLER) 
                {
                    DEBUGMSG(ZONE_IOCTL, (L"HPR: Exception in ioctl\r\n"));
                }
            }
            break;

        // gets the current device power state
        case IOCTL_POWER_GET: 
            DEBUGMSG(ZONE_IOCTL, (L"HPR: IOCTL_POWER_GET\r\n"));
            if (pBufOut != NULL && dwLenOut >= sizeof(CEDEVICE_POWER_STATE)) 
            {
                __try 
                {
                    *(PCEDEVICE_POWER_STATE)pBufOut = pDevice->powerState;
 
                    RetVal = TRUE;
                    DEBUGMSG(ZONE_IOCTL, (L"HPR: IOCTL_POWER_GET (D%u) \r\n", pDevice->powerState));
                }
                __except(EXCEPTION_EXECUTE_HANDLER) 
                {
                    DEBUGMSG(ZONE_IOCTL, (L"HPR:Exception in ioctl\r\n"));
                }
            }  

        case HPR_IOCTL_HPRF_SETGPIO:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Set GPIO.\r\n")));
            break;

        case HPR_IOCTL_HPRF_CLRGPIO:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Clear GPIO.\r\n")));
            break;

        case HPR_IOCTL_HPRF_GETGPIO:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Get GPIO.\r\n")));
            break;

        case HPR_IOCTL_POWER_ON:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Power On Command.\r\n")));
            break;
            
        case HPR_IOCTL_POWER_OFF:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Power Off Command\r\n")));
            break;

        case HPR_IOCTL_QUERY_POWER_STATE:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Query Power State\r\n")));
            break;

        // Acquire start
        case HPR_IOCTL_ACQUIRE_START:
            
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Start Acquire\r\n")));
            // If no timeout is supplied, set to default.
            if (pBufIn[0] == 0)
                u32AcquireTimeout = RFACQUIRE_PROCESS_DEFAULT_TIMEOUT;
            else
                u32AcquireTimeout = pBufIn[0];
            RFIntRFApp.pF_AcquireStart(u32AcquireTimeout);

            break;

        // Acquire stop
        case HPR_IOCTL_ACQUIRE_STOP:
            
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Stop Acquire\r\n")));
            RFIntRFApp.pF_AcquireStop(1);

            break;

        case HPR_IOCTL_SET_RF_ID:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Set RF ID, data = %d\r\n"), pBufIn[0]));
			rfparam.RFID = pBufIn[0];
            break;

        case HPR_IOCTL_GET_RF_ID:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Get RF ID\r\n")));
            pBufOut[0] = rfparam.RFID;
            dwLenOut = sizeof (pBufOut);
            *pdwActualOut = 1; // 1 byte
            break;

        case HPR_IOCTL_SET_RF_CHANNEL:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Set RF Channel, data = %d\r\n"), pBufIn[0]));
            rfparam.channel = pBufIn[0];
            // Call the driver function to set the channel
            RFIntRFApp.pF_SetChannel(pBufIn[0]);
            break;

        case HPR_IOCTL_GET_RF_CHANNEL:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Get RF Channel\r\n")));
            pBufOut[0] = rfparam.channel;
            dwLenOut = sizeof (pBufOut);
            *pdwActualOut = 1; // 1 byte

            break;

        case HPR_IOCTL_SET_RF_POWER_HI_LOW:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Set RF Power HiLow\r\n")));
            rfparam.powerhilo = pBufIn[0];
            // Set it in the driver now
            RFIntRFApp.pF_SetPowerLevel((UINT16)pBufIn[0]);
            break;

        case HPR_IOCTL_GET_RF_POWER_HI_LOW:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Get RF Power HiLow\r\n")));
            pBufOut[0] = rfparam.powerhilo;
            dwLenOut = sizeof (pBufOut);
            *pdwActualOut = 1; // 1 byte
            break;

        case HPR_IOCTL_RF_APP_INIT_DONE:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - RF App init done\r\n")));
            // Set the global variable here
            pHPRDevice->bApplicationInitDone = TRUE;

			// TY: try to see if we can reconnect to RF gateway (after setting proper UID, TSID, etc.)
			if (rfparamlocal.UID_Valid == UID_SIGN)
			{
				//RFAPP_LinkRequestHandlerStop();
				RFIntRFApp.pF_LookupSettingsAndConnectDeviceToNetwork();    
			}
            break;

        case HPR_IOCTL_GET_GW_UID:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Get GW UID.\r\n")));
            // See if we have enough space in the buffer
            if (dwLenOut > (UID_MAX_SIZE-1))
            {
                memcpy(pBufOut, &rfparam.gatewayUID[0], UID_MAX_SIZE);
                *pdwActualOut = UID_MAX_SIZE;
            }
            else
            {
                pBufOut[0] = 0;
                *pdwActualOut = 0;
                DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Get GW UID Buffer size insufficient.\r\n")));
            }
            break;

        case HPR_IOCTL_GET_GW_UID_STRING:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Get GW UID String.\r\n")));
            // See if we have enough space in the buffer
            strncpy((char*)pBufOut, (char *)rfparamlocal.gatewayUIDString, dwLenOut);
            *pdwActualOut = dwLenOut;
            break;


            break;

        case HPR_IOCTL_SET_GW_UID:
            // This setting needs a reboot
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Set GW UID.\r\n")));
            if (dwLenIn > (UID_MAX_SIZE-1))
            {
                memcpy(&rfparam.gatewayUID[0], pBufIn, UID_MAX_SIZE);
                ConvertUIDtoString((char*)rfparam.gatewayUID, rfparamlocal.gatewayUIDString, gAddrModeLong_c);
                // Inform the driver (needed when the restore is done from the INI file after a OS update)
                RFIntRFApp.pF_SetGatewayUID_PANID_ADDRMODE(TRUE, &rfparam.gatewayUID[0], rfparam.gatewayPanID, rfparam.addrMode); 
            }
            else
            {
                DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Set GW UID Buffer size invalid.\r\n")));
            }
            break;

        case HPR_IOCTL_GET_UID:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Get UID.\r\n")));
            *pdwActualOut = 0;      // Set error state
            // See if we have enough space in the buffer
            if (dwLenOut > (UID_MAX_SIZE-1))
            {
                if (rfparamlocal.UID_Valid == UID_SIGN)
                {
                    memcpy(pBufOut, &rfparam.UID[0], UID_MAX_SIZE);
                    *pdwActualOut = UID_MAX_SIZE;
                }
                else
                {
                    memset(pBufOut, 0, UID_MAX_SIZE);
                }
            }
            else
            {
                DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Get UID Buf Len insufficient.\r\n")));
            }
            break;

        case HPR_IOCTL_GET_UID_STRING:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Get UID String.\r\n")));
            // See if we have enough space in the buffer
            strncpy((char*)pBufOut, (char*)rfparamlocal.UIDString, dwLenOut);
            // If the stored UID is valid, send actual count, else set to 0 for the application to know
            if (rfparamlocal.UID_Valid == UID_SIGN)
                *pdwActualOut = dwLenOut;
            else
                *pdwActualOut = 0;
            break;

        case HPR_IOCTL_SET_UID:
            // This setting needs a reboot.
            // Note: This is stored in flash EEP section. Registry entry is for human read-only purpose.
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Set UID.\r\n")));

            // Set error
            *pdwActualOut = 0;

            // Check the buffer size
            if (dwLenIn < UID_MAX_SIZE)
            {
                DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Set UID Buffer size invalid.\r\n")));
                break;
            }

            // Proceed if buffer-in size is alright. First update the registry key (used for human-read only).
            // Update the rf param variable and update the string variable
            memcpy(&rfparam.UID[0], pBufIn, UID_MAX_SIZE);
            ConvertUIDtoString(rfparam.UID, rfparamlocal.UIDString, gAddrModeLong_c);
            // Update the driver
            RFIntRFApp.pF_SetLocaldeviceUID_PANID(TRUE, &rfparam.UID[0], rfparam.panID);

            // Save it in Flash EEPROM section.
            // Always do read-modify-write to make sure not to corrupt other parameters
            HPRF_ReadWriteEEPROM(&eep_param, 1);
            // Validate the UID and sign (EEP and local) and write.
            memcpy(eep_param.RfUID, rfparam.UID, UID_MAX_SIZE);
            eep_param.RfUIDSign = UID_SIGN;
            rfparamlocal.UID_Valid = UID_SIGN;
            // Now update the EEPROM image, and if the write is success, return SUCCESS
            if (HPRF_ReadWriteEEPROM(&eep_param, 0))
            {
                *pdwActualOut = 1;
            }
            break;

        case HPR_IOCTL_GW_CONN_QUERY:
            //DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - GW Conn. status query\r\n")));
            pBufOut[0] = (BYTE)RFIntRFApp.pF_GatewayConnectionStatusQuery();
            *pdwActualOut = 1;
            break;

        case HPR_IOCTL_CS_CONN_QUERY:
            //DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Ctrl sys Conn status query.\r\n")));
            pBufOut[0] = (BYTE)RFIntRFApp.pF_ControlSystemConnectionStatusQuery();
            *pdwActualOut = 1;
            break;

        case HPR_IOCTL_GW_ACQUIRED_QUERY:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - GW Acquired query.\r\n")));
            for (dwTemp = 0; dwTemp < UID_MAX_SIZE; dwTemp++)
            {
                if (rfparam.gatewayUID[dwTemp] != 0)
                    break;
            }
            if (dwTemp == UID_MAX_SIZE)
                pBufOut[0] = 0; // No gateway acquired
            else
                pBufOut[0] = 1; // Gateway acquired
            break;

        case HPR_IOCTL_GET_GW_NAME:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Get GW Name.\r\n")));
            dwTemp = strlen(rfparamlocal.gatewayName);
            if (dwLenOut > dwTemp)
            {
                strcpy(pBufOut, rfparamlocal.gatewayName);
                *pdwActualOut = dwTemp;
            }
            else
            {
                pBufOut[0] = 0;
                *pdwActualOut = 0;
                DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Get GW Name Buf Len insufficient.\r\n")));
            }
            break;
            
        case HPR_IOCTL_GET_SIGNAL_STRENGTH:
            //DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Get Signal Strength.\r\n")));
            if (dwLenOut < sizeof (DWORD))
            {
                DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Get Signal Strength Buff Len insufficient.\r\n")));
            }
            else
            {
                if (!g_bRF_LQI_Active)
                {
                    RFIntRFApp.pF_GetLQI (0, TRUE, &lq);
                    lq_norm= RFIntRFApp.pF_ConvertLQIToIEEEScaleWord(lq, LQI_DISP_OFFSET);
                    *(DWORD*)pBufOut = (DWORD)lq_norm;
                    //DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Get Signal Strength: %d\r\n"), lq_norm));
                }
                else
                {
                    *(DWORD*)pBufOut = 0;
                }
            }
            break;

        case HPR_IOCTL_LQI_START:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - LQI Start.\r\n")));
            /* Always default the values when the tasks tarts */
            LocalDevLQI = 255;
            RemoteLQI = 255;
            g_bRF_LQI_Active = TRUE;
            // Set the event so that the thread starts
            SetEvent(pHPRDevice->hLQIEvent);
            
            break;

        case HPR_IOCTL_LQI_STOP:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - LQI Stop.\r\n")));
            // Reset the event such that the thread goes to sleep
            ResetEvent(pHPRDevice->hLQIEvent);
            g_bRF_LQI_Active = FALSE;

            break;

        case HPR_IOCTL_GET_RF_PWR_BOOST:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Get RF Power Boost\r\n")));
            *(DWORD*)pBufOut = (DWORD)rfparam.powerBoost;
            dwLenOut = sizeof (pBufOut);
            *pdwActualOut = 1; // 1 byte
            break;

        case HPR_IOCTL_SET_RF_PWR_BOOST:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Set RF Power Boost\r\n")));
            dwData = *(DWORD*) pBufIn;  // dword given
            if (dwData > 21)
            {
                // Limit the value if it exceeds
                dwData = 21;
            }
            rfparam.powerBoost = (UINT8) dwData;
            // Update the main variable
            gPowerBoost = rfparam.powerBoost;

            // Just like TPS-6X
            if (RFIntRFApp.pF_RFReceive_GetLocalDevicePower() >= RF_POWER_HI)
            {
                RFIntRFApp.pF_SetPowerLevel (RF_POWER_HI);
            }
            else
            {
                RFIntRFApp.pF_SetPowerLevel (RF_POWER_LO);
            }
            break;

        case HPR_IOCTL_GET_RF_DEBUG:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Get RF Debug\r\n")));
            dwData = 0;
            if (rfparam.RFTxDebug)
                dwData |= RF_RFTX_DEBUG_FLAG;
            if (rfparam.RFRxDebug)
                dwData |= RF_RFRX_DEBUG_FLAG;
            if (rfparam.RFGenericDebug)
                dwData |= RF_GENERIC_DEBUG_FLAG;
            // Fill the buffer with data
            *(DWORD*)pBufOut = dwData;              
            dwLenOut = sizeof (pBufOut);
            *pdwActualOut = 1; // 1 byte
            break;

        case HPR_IOCTL_SET_RF_DEBUG:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Set RF Debug\r\n")));
            
            // Check for RFTX debug settings
            dwData = *(DWORD*) &pBufIn[0];   // Control code
            dwTemp = *(DWORD*) &pBufIn[4];   // On off

            // Safety check
            if (dwTemp > 1)
                dwTemp = 1;

      
            // Update each individual settings
            // First RFTX debug
            if (dwData == RF_RFTX_DEBUG_FLAG || dwData == RF_ALL_DEBUG_FLAG)
            {
                rfparam.RFTxDebug = (UINT8)dwTemp;
            }
            
            // Now RFRX debug
            if (dwData == RF_RFRX_DEBUG_FLAG || dwData == RF_ALL_DEBUG_FLAG)
            {
                rfparam.RFRxDebug = (UINT8)dwTemp;
            }
            // Now RF Generic debug
            if (dwData == RF_GENERIC_DEBUG_FLAG || dwData == RF_ALL_DEBUG_FLAG)
            {
                rfparam.RFGenericDebug = (UINT8)dwTemp;
            }
            break;

        case HPR_IOCTL_GET_PAN_ID:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Get Pan ID\r\n")));
            *(DWORD*)pBufOut = (DWORD)rfparam.panID;
            dwLenOut = sizeof (pBufOut);
            *pdwActualOut = 1; // 1 byte
            break;

        case HPR_IOCTL_SET_PAN_ID:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Set Pan ID\r\n")));
            dwData = *(DWORD*) &pBufIn[0];  // dword given
            rfparam.panID = (UINT16)dwData;
            // Inform the driver (Need when restoring old settings from INI file, in case of OS upgrade)
            RFIntRFApp.pF_SetLocaldeviceUID_PANID(TRUE, &rfparam.UID[0], rfparam.panID); 
            break;

        case HPR_IOCTL_GET_GATEWAY_PAN_ID:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Get Gateway Pan ID\r\n")));
            *(DWORD*)pBufOut = (DWORD)rfparam.gatewayPanID;
            dwLenOut = sizeof (pBufOut);
            *pdwActualOut = 1; // 1 byte
            break;

        case HPR_IOCTL_SET_GATEWAY_PAN_ID:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Set Gateway Pan ID\r\n")));
            dwData = *(DWORD*) &pBufIn[0];  // dword given
            rfparam.gatewayPanID = (UINT16)dwData;
            // Inform the driver (Need when restoring old settings from INI file, in case of OS upgrade)
            RFIntRFApp.pF_SetGatewayUID_PANID_ADDRMODE(TRUE, &rfparam.gatewayUID[0], rfparam.gatewayPanID, rfparam.addrMode); 
            break;

        case HPR_IOCTL_GET_CCA_THRESHOLD:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Get CCA Threshold\r\n")));
            *(DWORD*)pBufOut = (DWORD)rfparam.CCAEnergyDetectThr;
            dwLenOut = sizeof (pBufOut);
            *pdwActualOut = 1; // 1 byte
            break;

        case HPR_IOCTL_SET_CCA_THRESHOLD:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Set CCA Threshold\r\n")));
            dwData = *(DWORD*) &pBufIn[0];  // dword given
            rfparam.CCAEnergyDetectThr = (UINT8)dwData;
            // Inform the driver (Need when restoring old settings from INI file, in case of OS upgrade)
            RFIntRFApp.pF_SetCCAEnergyDetectThr(rfparam.CCAEnergyDetectThr);
            break;

        case HPR_IOCTL_GET_RSSI_COMP_OFFSET:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Get RSSI Comp Offset\r\n")));
            *(DWORD*)pBufOut = (DWORD)rfparam.RSSIPowerCompOffset;
            dwLenOut = sizeof (pBufOut);
            *pdwActualOut = 1; // 1 byte
            break;

        case HPR_IOCTL_SET_RSSI_COMP_OFFSET:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Set RSSI comp offset\r\n")));
            dwData = *(DWORD*) &pBufIn[0];  // dword given
            rfparam.RSSIPowerCompOffset = (UINT8)dwData;
            // Inform the driver (Need when restoring old settings from INI file, in case of OS upgrade)
            RFIntRFApp.pF_SetRSSIPowerCompOffset(rfparam.RSSIPowerCompOffset);
            break;

        case HPR_IOCTL_GET_RF_RETRY:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Get Retry Count\r\n")));
            *(DWORD*)pBufOut = (DWORD)rfparam.retry;
            dwLenOut = sizeof (pBufOut);
            *pdwActualOut = 1; // 1 byte
            break;

        case HPR_IOCTL_SET_RF_RETRY:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Set RF Retry\r\n")));
            dwData = *(DWORD*) &pBufIn[0];  // dowrd given
            rfparam.retry = (UINT8)dwData;
            // Inform the driver (Need when restoring old settings from INI file, in case of OS upgrade)
            RFIntRFApp.pF_SetRetryCount(rfparam.retry);
            break;

        case HPR_IOCTL_GET_RETRY_COUNT_ON_CONN:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Get Retry Count on Connection\r\n")));
            *(DWORD*)pBufOut = (DWORD)rfparam.txPktretryonconnection;
            dwLenOut = sizeof (pBufOut);
            *pdwActualOut = 1; // 1 byte
            break;

        case HPR_IOCTL_SET_RETRY_COUNT_ON_CONN:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Set Retry count on connection\r\n")));
            dwData = *(DWORD*) &pBufIn[0];  // dword given
            rfparam.txPktretryonconnection = (UINT8)dwData;
            // Inform the driver (Need when restoring old settings from INI file, in case of OS upgrade)
            RFIntRFApp.pF_SetRetryCountOnConnectionReq(rfparam.txPktretryonconnection);
            break;

        case HPR_IOCTL_GET_PRE_LINK_COUNT:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Get Pre link count\r\n")));
            *(DWORD*)pBufOut = (DWORD)rfparam.PreLinkQueryPacketCnt;
            dwLenOut = sizeof (pBufOut);
            *pdwActualOut = 1; // 1 byte
            break;

        case HPR_IOCTL_SET_PRE_LINK_COUNT:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Set Pre link count\r\n")));
            dwData = *(DWORD*) &pBufIn[0];  // dword given
            rfparam.PreLinkQueryPacketCnt = (UINT8)dwData;
            // Inform the driver (Need when restoring old settings from INI file, in case of OS upgrade)
            RFIntRFApp.pF_SetPreLinkQueryPacketCnt(rfparam.PreLinkQueryPacketCnt);
            break;

        case HPR_IOCTL_UPDATE_RADIO_DATA:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Update TSID\r\n")));
            dwData = *(DWORD*) &pBufIn[0];  // dword given
            rfparamlocal.TSID = dwData;
            if (strlen(&pBufIn[4]))
            {
                // update the boot id message
                strcpy(bootidmsg, &pBufIn[4]);
            }
            else
            {
                DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Update TSID: NULL String\r\n")));
            }
            break;

        case HPR_IOCTL_RF_TESTS:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - RF Tests\r\n")));
            *pdwActualOut = RF_Tests(pBufIn, dwLenIn, pBufOut, dwLenOut);
            break;

        case HPR_IOCTL_DIAGNOSTICS:
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - RF Diagnostics.\r\n")));
            *pdwActualOut = RF_Diagnostics(pBufIn, dwLenIn, pBufOut, dwLenOut);
            break;

        default:
            DEBUGMSG(ZONE_IOCTL, (L"HPRF.DLL: Unknown IOCTL call: %d.\r\n", dwCode));
            break;
    }
    // Return a result code to the user, which they can fetch with GetLastError().
    SetLastError(dwReturn);
    return RetVal;
}


// These are the stubs for a COM port like application:  Read, Write, and Seek.  In our
// driver they are not used, but included here as stubs as an example of their implementation.

DWORD
HPR_Read (
    DWORD hOpenContext,
    LPVOID pBuffer,
    DWORD Count )
{
    DEBUGMSG(ZONE_FUNCTION, (TEXT("HPRF.DLL: +HPR_Read\r\n")));
    DEBUGMSG(ZONE_FUNCTION, (TEXT("HPRF.DLL: -HPR_Read\r\n")));
    return 1;
}

DWORD
HPR_Write (
    DWORD hOpenContext,
    LPCVOID pSourceBytes,
    DWORD NumberOfBytes )
{
    DEBUGMSG(ZONE_FUNCTION, (TEXT("HPRF.DLL: +HPR_Write\r\n")));
    DEBUGMSG(ZONE_FUNCTION, (TEXT("HPRF.DLL: -HPR_Write\r\n")));
    return 1;
}

DWORD
HPR_Seek (
    DWORD hOpenContext,
    long Amount,
    WORD Type )
{
    DEBUGMSG(ZONE_FUNCTION, (TEXT("HPRF.DLL: +HPR_Seek\r\n")));
    DEBUGMSG(ZONE_FUNCTION, (TEXT("HPRF.DLL: -HPR_Seek\r\n")));
    return -1;
}

// Load DStore parameters to local variables
void HPRF_LoadDStoreParams(void)
{
	HANDLE hDriver;
    DWORD dwRet;
	DWORD dwStatus;

    UINT8 temp_buf[16];
    DWORD dwSize = sizeof(DWORD);
	int	i;

	ER_RADIO_SETTINGS_STRUCT erRadioSettings;

    // For EEPROM read
    // HANDLE hDriver;
    // DWORD dwBytesReturned;
    
    // Create instance for EEPROM params
    EEP_PARAMS eep_param;

    // Clear the temporary buffer
    memset(temp_buf, 0, sizeof (temp_buf));

    hDriver  = CreateFile(TEXT("PDS1:"),GENERIC_READ|GENERIC_WRITE,FILE_SHARE_READ|FILE_SHARE_WRITE,NULL,OPEN_EXISTING,0,NULL);
    if (hDriver == INVALID_HANDLE_VALUE)
    {
        RETAILMSG(1, (TEXT("\r\nERROR HPRF_ReadWriteEEPROM: - Failed to open PDS1 handle.\r\n")));
        goto CleanUp;
    }

	if(!(SetFilePointer(hDriver,0,NULL,FILE_END) >= SYSTEM_STORAGE_SIZE))
		RETAILMSG(1, (TEXT("\r\nERROR HPRF_ReadWriteEEPROM: - Invalid Size.\r\n")));
        


    // Check the User EEPORM signature
	// If signature is invalid, load defaults
    if (!VerifyUserEepParams())
	{
		
        RETAILMSG(1, (TEXT("\r\nERROR HPRF Init: VerifyUserEepParams Fail! using defaults\r\n")));
		memset(&rfparam, 0, sizeof (RF_PARAM));
		
		erRadioSettings.ulER_PanID = 0xFEED;
		rfparam.panID = (UINT16)erRadioSettings.ulER_PanID;
    
		erRadioSettings.ulER_ShortAddr = 0xFFFE;
		rfparam.shortAddr = (UINT16)erRadioSettings.ulER_ShortAddr;

		erRadioSettings.ulER_AddrMode = 0x03;
		rfparam.addrMode = (UINT8)erRadioSettings.ulER_AddrMode;

		erRadioSettings.ulER_CCAEnergyDetectThr = 0x96; 
		rfparam.CCAEnergyDetectThr = (UINT8)erRadioSettings.ulER_CCAEnergyDetectThr;
		//RETAILMSG(1, (TEXT("\r\nrfparam.CCAEnergyDetectThr set: %x\r\n"),rfparam.CCAEnergyDetectThr));

		erRadioSettings.ulER_RSSIPowerCompOffset = 0xB5;
		rfparam.RSSIPowerCompOffset = (UINT8)erRadioSettings.ulER_RSSIPowerCompOffset;

		erRadioSettings.ulER_Channel = 0x14;
		rfparam.channel = (UINT8)erRadioSettings.ulER_Channel;

		erRadioSettings.ulER_Power = 0xFFFF;
		rfparam.power = (UINT16)erRadioSettings.ulER_Power;

		erRadioSettings.ulER_Retry = 0x09;
		rfparam.retry = (UINT8)erRadioSettings.ulER_Retry;

		memset(&erRadioSettings.ulER_RetryDelay, 0, sizeof (erRadioSettings.ulER_RetryDelay));
		memcpy(&rfparam.retrydelay, &erRadioSettings.ulER_RetryDelay, RF_MAX_NO_OF_RETRIES_DELAYS);

		erRadioSettings.ulER_ResultcodeTimeout = 0x0003;
		rfparam.resultcodetimeout = (UINT16)erRadioSettings.ulER_ResultcodeTimeout;

		erRadioSettings.ulER_Heartbeat = 0x04;
		rfparam.heartBeat = (UINT8)erRadioSettings.ulER_Heartbeat;

		memset(&erRadioSettings.ulER_LinkRetryDelay, 0, sizeof (erRadioSettings.ulER_LinkRetryDelay));
		memcpy(&rfparam.link_retrydelay, &erRadioSettings.ulER_LinkRetryDelay, RF_MAX_NO_OF_RETRIES_DELAYS);

		memcpy(&rfparam.gatewayUID, &erRadioSettings.ucER_GatewayUID, UID_MAX_SIZE);
		ConvertUIDtoString((char*)rfparam.gatewayUID, rfparamlocal.gatewayUIDString, gAddrModeLong_c);

		erRadioSettings.ulER_GatewayPanID = 0xFEED;
		rfparam.gatewayPanID = (UINT16)erRadioSettings.ulER_GatewayPanID;

		erRadioSettings.ulER_RFID = 0x03;
		rfparam.RFID = (UINT8)erRadioSettings.ulER_RFID;

		erRadioSettings.ulER_RFTxDebug = 0x00;
		rfparam.RFTxDebug = (UINT8)erRadioSettings.ulER_RFTxDebug;

		erRadioSettings.ulER_RFRxDebug = 0x00;
		rfparam.RFRxDebug = (UINT8)erRadioSettings.ulER_RFRxDebug;

		erRadioSettings.ulER_RFGenericDebug = 0x00;
		rfparam.RFGenericDebug = (UINT8)erRadioSettings.ulER_RFGenericDebug;

		erRadioSettings.ulER_PowerHiLo = 0x01;
		rfparam.powerhilo = (UINT8)erRadioSettings.ulER_PowerHiLo;

		erRadioSettings.ulER_LinkLQIThreshold = 0xB4;
		rfparam.LinkLQIThreshold = (UINT8)erRadioSettings.ulER_LinkLQIThreshold;

		erRadioSettings.ulER_TxPktretryonconnection = 0x03;
		rfparam.txPktretryonconnection = (UINT8)erRadioSettings.ulER_TxPktretryonconnection;

		erRadioSettings.ulER_PreLinkQueryPacketCnt = 0x01;
		rfparam.PreLinkQueryPacketCnt = (UINT8)erRadioSettings.ulER_PreLinkQueryPacketCnt;

		erRadioSettings.ulER_PowerBoost = 0x00;
		rfparam.powerBoost = (UINT8)erRadioSettings.ulER_PowerBoost;

		SetFilePointer(hDriver,DS_ER_RADIO_SETTINGS,NULL,FILE_BEGIN);
		WriteFile(hDriver,&erRadioSettings,sizeof(ER_RADIO_SETTINGS_STRUCT),&dwRet,NULL);
		if(dwRet != sizeof(ER_RADIO_SETTINGS_STRUCT))
			RETAILMSG(1, (TEXT("\r\nERROR HPRF Write DataStore Error\r\n")));

		goto CleanUp;
	}
    // Read the EEPROM param to RF global at startup
    HPRF_ReadWriteEEPROM(&eep_param, 1);

    if (eep_param.RfUIDSign == UID_SIGN)
    {
        rfparamlocal.UID_Valid = UID_SIGN;
        // Only if UID is valid, else set to zero.
        memcpy(&rfparam.UID[0], eep_param.RfUID, UID_MAX_SIZE);
    }
	else 
	{
		RETAILMSG(1, (TEXT("\r\nFound invalid signature, restoring defaults\r\n")));
		memset(&erRadioSettings, 0, sizeof(erRadioSettings));
		
		erRadioSettings.ulER_PanID = 0xFEED;
		erRadioSettings.ulER_ShortAddr = 0xFFFE;
		erRadioSettings.ulER_AddrMode = 0x03;
		erRadioSettings.ulER_CCAEnergyDetectThr = 0x96; 
		erRadioSettings.ulER_RSSIPowerCompOffset = 0xB5;
		erRadioSettings.ulER_Channel = 0x14;
		erRadioSettings.ulER_Power = 0xFFFF;
		erRadioSettings.ulER_Retry = 0x09;
		erRadioSettings.ulER_ResultcodeTimeout = 0x0003;
		erRadioSettings.ulER_Heartbeat = 0x04;
		erRadioSettings.ulER_GatewayPanID = 0xFEED;
		erRadioSettings.ulER_RFID = 0x03;
		erRadioSettings.ulER_RFTxDebug = 0x00;
		erRadioSettings.ulER_RFRxDebug = 0x00;
		erRadioSettings.ulER_RFGenericDebug = 0x00;
		erRadioSettings.ulER_PowerHiLo = 0x01;
		erRadioSettings.ulER_LinkLQIThreshold = 0xB4;
		erRadioSettings.ulER_TxPktretryonconnection = 0x03;
		erRadioSettings.ulER_PreLinkQueryPacketCnt = 0x01;
		erRadioSettings.ulER_PowerBoost = 0x00;
		
		SetFilePointer(hDriver,DS_ER_RADIO_SETTINGS,NULL,FILE_BEGIN);
		WriteFile(hDriver,&erRadioSettings,sizeof(ER_RADIO_SETTINGS_STRUCT),&dwRet,NULL);

        rfparamlocal.UID_Valid = 0;
        memset(&rfparam.UID[0], 0, UID_MAX_SIZE);
	}
    // In anyways, convert to string and store it.
    ConvertUIDtoString(rfparam.UID, rfparamlocal.UIDString, gAddrModeLong_c);

    // Now read all settings from the registry
    // NOTE: Timing observation
    // RegOpenKeyEx takes about 80uS, and reading all keys takes about 330uS
	SetFilePointer(hDriver,DS_ER_RADIO_SETTINGS,NULL,FILE_BEGIN);
	ReadFile(hDriver,&erRadioSettings,sizeof(ER_RADIO_SETTINGS_STRUCT),&dwRet,NULL);
	if(dwRet != sizeof(ER_RADIO_SETTINGS_STRUCT))
		RETAILMSG(1, (TEXT("\r\nERROR HPRF Read DataStore Error\r\n")));

    rfparam.panID = (UINT16)erRadioSettings.ulER_PanID;
    
    rfparam.shortAddr = (UINT16)erRadioSettings.ulER_ShortAddr;

    rfparam.addrMode = (UINT8)erRadioSettings.ulER_AddrMode;

	rfparam.CCAEnergyDetectThr = (UINT8)erRadioSettings.ulER_CCAEnergyDetectThr;

    rfparam.RSSIPowerCompOffset = (UINT8)erRadioSettings.ulER_RSSIPowerCompOffset;

    rfparam.channel = (UINT8)erRadioSettings.ulER_Channel;

    rfparam.power = (UINT16)erRadioSettings.ulER_Power;

    rfparam.retry = (UINT8)erRadioSettings.ulER_Retry;

    memcpy(&rfparam.retrydelay, &erRadioSettings.ulER_RetryDelay, RF_MAX_NO_OF_RETRIES_DELAYS);

    rfparam.resultcodetimeout = (UINT16)erRadioSettings.ulER_ResultcodeTimeout;

    rfparam.heartBeat = (UINT8)erRadioSettings.ulER_Heartbeat;

    memcpy(&rfparam.link_retrydelay, &erRadioSettings.ulER_LinkRetryDelay, RF_MAX_NO_OF_RETRIES_DELAYS);

    memcpy(&rfparam.gatewayUID, &erRadioSettings.ucER_GatewayUID, UID_MAX_SIZE);
    ConvertUIDtoString((char*)rfparam.gatewayUID, rfparamlocal.gatewayUIDString, gAddrModeLong_c);

    rfparam.gatewayPanID = (UINT16)erRadioSettings.ulER_GatewayPanID;

    rfparam.RFID = (UINT8)erRadioSettings.ulER_RFID;

    rfparam.RFTxDebug = (UINT8)erRadioSettings.ulER_RFTxDebug;

    rfparam.RFRxDebug = (UINT8)erRadioSettings.ulER_RFRxDebug;

    rfparam.RFGenericDebug = (UINT8)erRadioSettings.ulER_RFGenericDebug;

    rfparam.powerhilo = (UINT8)erRadioSettings.ulER_PowerHiLo;

    rfparam.LinkLQIThreshold = (UINT8)erRadioSettings.ulER_LinkLQIThreshold;

    rfparam.txPktretryonconnection = (UINT8)erRadioSettings.ulER_TxPktretryonconnection;

    rfparam.PreLinkQueryPacketCnt = (UINT8)erRadioSettings.ulER_PreLinkQueryPacketCnt;

    rfparam.powerBoost = (UINT8)erRadioSettings.ulER_PowerBoost;

#if 0	// TY: enabled for debugging
    RETAILMSG(1, (TEXT("---------------Printing------------------")));
    
    RETAILMSG(1, (L"\r\nUID: "));
    for (i = 0; i < 8; i++)
        RETAILMSG(1, (L"%02x ", rfparam.UID[i]));

    RETAILMSG(1, (L"\r\nPanID: 0x%x", rfparam.panID));
    RETAILMSG(1, (L"\r\nShortAddr: 0x%x", rfparam.shortAddr));
    RETAILMSG(1, (L"\r\nAddr Mode: 0x%x", rfparam.addrMode));
    RETAILMSG(1, (L"\r\nCCAEnergyDetectThr: 0x%x", rfparam.CCAEnergyDetectThr));
    RETAILMSG(1, (L"\r\nRSSIPowerCompOffset: 0x%x", rfparam.RSSIPowerCompOffset));
    RETAILMSG(1, (L"\r\nChannel: 0x%x", rfparam.channel));
    RETAILMSG(1, (L"\r\npower: 0x%x", rfparam.power));
    RETAILMSG(1, (L"\r\nretry: 0x%x", rfparam.retry));
    
    RETAILMSG(1, (L"\r\nRetry Delay: "));
    for (i = 0; i < 10; i++)
        RETAILMSG(1, (L"%02x ", rfparam.retrydelay[i]));

    RETAILMSG(1, (L"\r\nresultcodetimeout: 0x%x", rfparam.resultcodetimeout));
    RETAILMSG(1, (L"\r\nheartBeat: 0x%x", rfparam.heartBeat));

    RETAILMSG(1, (L"\r\nlink_retrydelay: "));
    for (i = 0; i < 10; i++)
        RETAILMSG(1, (L"%02x ", rfparam.link_retrydelay[i]));

    RETAILMSG(1, (L"\r\nGatewayUID: "));
    for (i = 0; i < 8; i++)
        RETAILMSG(1, (L"%02x ", rfparam.gatewayUID[i]));

    RETAILMSG(1, (L"\r\ngatewayPanID: 0x%x", rfparam.gatewayPanID));
    RETAILMSG(1, (L"\r\nRFID: 0x%x", rfparam.RFID));
    RETAILMSG(1, (L"\r\nRFTxDebug: 0x%x", rfparam.RFTxDebug));
    RETAILMSG(1, (L"\r\nRFRxDebug: 0x%x", rfparam.RFRxDebug));
    RETAILMSG(1, (L"\r\nRFGenericDebug: 0x%x", rfparam.RFGenericDebug));
    RETAILMSG(1, (L"\r\npowerhilo: 0x%x", rfparam.powerhilo));
    RETAILMSG(1, (L"\r\nLinkLQIThreshold: 0x%x", rfparam.LinkLQIThreshold));
    RETAILMSG(1, (L"\r\ntxPktretryonconnection: 0x%x", rfparam.txPktretryonconnection));
    RETAILMSG(1, (L"\r\nPreLinkQueryPacketCnt: 0x%x", rfparam.PreLinkQueryPacketCnt));
    RETAILMSG(1, (L"\r\npowerBoost: 0x%x", rfparam.powerBoost));
#endif

CleanUp:
    // Don't forget to close the handle.
    if (hDriver != INVALID_HANDLE_VALUE)
        CloseHandle(hDriver);
    
}

// Send Control message to Application
DWORD SendCtrlMsgToApp(PVOID lpBuffer, DWORD cbDataSize)
{
    BOOL bWriteMsg = FALSE;
#if	0		// TY: debugging only
	int i;
	RETAILMSG(1, (TEXT("\r\nRF Ctrl Msg RX: [")));
	for (i = 0; i < cbDataSize; i++)
	{
		RETAILMSG(1, (TEXT("%02x "), ((UCHAR*)lpBuffer)[i]));
	}
	RETAILMSG(1, (TEXT("]\r\n")));
#endif

    // 20100518: Check for power state D3. We send the GW connect message in PowerUp function and then
    // we see GW and CTRL system disconnect message. If it reaches application, it seems to discard
    // the initial key press from sending to control system. Adding this check prevents that by not sending
    // disconnected messages to application when we are coming out of suspend mode.
    if (pHPRDevice->RF_Cntrl_Queue_write && pHPRDevice->bApplicationInitDone && pHPRDevice->powerState < D3)
    {
        bWriteMsg = WriteMsgQueue(pHPRDevice->RF_Cntrl_Queue_write, lpBuffer, cbDataSize, OS_NO_SUSPEND, 0);
        if (bWriteMsg == FALSE)
            RETAILMSG(1, (TEXT("HPRF SendCtrlMsgToApp Fail. Error: %d\r\n"), GetLastError()));
    }
    else
    {
        if (pHPRDevice->RF_Cntrl_Queue_write == NULL)
            RETAILMSG(1, (L"HPR SendCtrlMsgToApp: Invalid handle\r\n"));
    }
    return (DWORD)bWriteMsg;
}

// Send Cresent packet to Application
DWORD SendCresnetPacketToApp(PVOID lpBuffer, DWORD cbDataSize)
{
    BOOL bWriteMsg = FALSE;

#if	0		// TY: debugging only
	int i;
	RETAILMSG(1, (TEXT("\r\nRFRX Msg: [")));
	for (i = 0; i < cbDataSize; i++)
	{
		RETAILMSG(1, (TEXT("%02x "), ((UCHAR*)lpBuffer)[i]));
	}
	RETAILMSG(1, (TEXT("]\r\n")));
#endif

	if (pHPRDevice->RF_Rx_Queue_write && pHPRDevice->bApplicationInitDone)
    {
        // Timeout for 100ms
        bWriteMsg = WriteMsgQueue(pHPRDevice->RF_Rx_Queue_write, lpBuffer, cbDataSize, 100, 0);
        if (bWriteMsg == FALSE)
        {
            RETAILMSG(1, (TEXT("SendCresnetPacketToApp Fail. Error: %d\r\n"), GetLastError()));
        }
    }
    else
    {
        if (pHPRDevice->RF_Rx_Queue_write == NULL)
            RETAILMSG(1, (L"SendCresnetPacketToApp: Invalid handle\r\n"));
    }
    return (DWORD)bWriteMsg;
}


// Send Local Cresent packet to Application
DWORD SendLocalCresnetPacketToApp(PVOID lpBuffer, DWORD cbDataSize)
{
    BOOL bWriteMsg = FALSE;

#if	0		// TY: debugging only
	int i;
	RETAILMSG(1, (TEXT("\r\nRFRX2 Msg: [")));
	for (i = 0; i < cbDataSize; i++)
	{
		RETAILMSG(1, (TEXT("%02x "), ((UCHAR*)lpBuffer)[i]));
	}
	RETAILMSG(1, (TEXT("]\r\n")));
#endif

    if (pHPRDevice->RF_Rx_Queue_write && pHPRDevice->bApplicationInitDone)
    {
        // Timeout for 100ms
        bWriteMsg = WriteMsgQueue(pHPRDevice->RF_Rx_Queue_write, lpBuffer, cbDataSize, 100, 0);
        if (bWriteMsg == FALSE)
        {
            RETAILMSG(1, (TEXT("SendLocalCresnetPacketToApp Fail. Error: %d\r\n"), GetLastError()));
        }
    }
    else
    {
        if (pHPRDevice->RF_Rx_Queue_write == NULL)
            RETAILMSG(1, (L"SendLocalCresnetPacketToApp: Invalid handle\r\n"));
    }
    return (DWORD)bWriteMsg;
}

// Send Diagnostics and test data messages to Application
DWORD SendDiagDataToApp(PVOID lpBuffer, DWORD cbDataSize)
{
    BOOL bWriteMsg = 0;
    // Check the app init done and then send it to the application.
    if (pHPRDevice->RF_Diag_Queue_write && pHPRDevice->bApplicationInitDone)
    {
        bWriteMsg = WriteMsgQueue(pHPRDevice->RF_Diag_Queue_write, lpBuffer, cbDataSize, OS_NO_SUSPEND, 0);
        if (bWriteMsg == FALSE)
            RETAILMSG(1, (TEXT("HPRF SendDiagDataToApp Fail. Error: %d\r\n"), GetLastError()));
    }
    else
    {
        if (pHPRDevice->RF_Diag_Queue_write == NULL)
            RETAILMSG(1, (L"HPR SendDiagDataToApp: Invalid handle\r\n"));
    }
    return (DWORD)bWriteMsg;
}

// Thread waits on Transmit pipe.
// Whenever application writes something to this pipe, it will wake up.
// Cresnet Packet Queue thread
DWORD RFTx_CresnetPacketThread(void * pContext)
{
    UCHAR tx_buf[260];
    hprf_hw_dev *pDevice = (hprf_hw_dev*)pContext;
    DWORD dwLen, dwFlags;

    //////////////////////////////////////////////////////
    // Wait for initialization to complete

    RFIntHostApp.pF_StartupWait();
    //	Wait for RF Stack to be intialized
    while (!RFReceive_IsRFStackInitialized())	{	RFIntHostApp.pF_HwDelayMsec(100);	}
    
    RFIntHostApp.pF_ErrorPrintf("\r\nRFTx_CresnetPacketThread: Running!");

    //////////////////////////////////////////////////////
    // Start the thread
    while (TRUE)
    {
        if (TRUE == ReadMsgQueue(pDevice->RF_Tx_Queue_read, (void*)tx_buf, sizeof (tx_buf), &dwLen, OS_SUSPEND, &dwFlags))
        {
			// Fill in 1st byte with correct RFID:
			tx_buf[0] = rfparam.RFID;

#if	0		// PRASAD debug only
            RETAILMSG(1, (TEXT("\r\nRFTX Msg: [")));
            for (dwFlags = 0; dwFlags < dwLen; dwFlags++)
            {
                RETAILMSG(1, (TEXT("%02x "), tx_buf[dwFlags]));
            }
            RETAILMSG(1, (TEXT("]\r\n")));
#endif
            // Post to pipe
            RFIntRFApp.pF_PostCresnetPacket((unsigned char *)tx_buf, (unsigned short)dwLen, NULL, 0, 0);

        }
        else
        {
            RETAILMSG(1, (TEXT("RFTx_CresnetPacketThread: ReadMsgQueue Fail: %d\r\n"), GetLastError()));
        }
    }
    return 0;
}


// Cresnet Packet Queue thread
DWORD RFTx_LocalCresnetPacketThread(void * pContext)
{
    UCHAR tx_buf[260];
    hprf_hw_dev *pDevice = (hprf_hw_dev*)pContext;
    DWORD dwLen, dwFlags;

    //////////////////////////////////////////////////////
    // Wait for initialization to complete

    RFIntHostApp.pF_StartupWait();
    //	Wait for RF Stack to be intialized
    while (!RFReceive_IsRFStackInitialized())	{	RFIntHostApp.pF_HwDelayMsec(100);	}
    
    RFIntHostApp.pF_ErrorPrintf("\r\nRFTx_LocalCresnetPacketThread: Running!");

    //////////////////////////////////////////////////////
    // Start the thread
    while (TRUE)
    {
        if (TRUE == ReadMsgQueue(pDevice->RF_Tx_Queue2_read, (void*)tx_buf, sizeof (tx_buf), &dwLen, OS_SUSPEND, &dwFlags))
        {
			// Fill in 1st byte with correct RFID:
			tx_buf[0] = rfparam.RFID;

#if	0		// PRASAD debug only
            RETAILMSG(1, (TEXT("\r\nRFTX2 Msg: [")));
            for (dwFlags = 0; dwFlags < dwLen; dwFlags++)
            {
                RETAILMSG(1, (TEXT("%02x "), tx_buf[dwFlags]));
            }
            RETAILMSG(1, (TEXT("]\r\n")));
#endif
            // Post to pipe
            RFIntRFApp.pF_PostLocalCresnetPacket((unsigned char *)tx_buf, (unsigned short)dwLen, NULL, 0, 0);

        }
        else
        {
            RETAILMSG(1, (TEXT("RFTx_LocalCresnetPacketThread: ReadMsgQueue Fail: %d\r\n"), GetLastError()));
        }
    }
    return 0;
}

///////////////////////////////////////////////
/// LQI related functions
INT32 RF_LQI_Timer_EchoRequest_CB(TYPE_RF_ECHO_REPLY_PACKET *pechoreply)
{
    LocalDevLQI = pechoreply->localrxlqi;
    RemoteLQI   = pechoreply->remoterxlqi;
    return (SUCCESS);
}

INT32 RF_LQI_Timer_NetworkService_CB(TYPE_RF_DISCOVER_REPLY *pRFDis, BOOL ignore)
{
    LocalDevLQI = pRFDis->linkqualityatlocaldevice;
    RemoteLQI   = pRFDis->disc.linkquality;
    return (SUCCESS);
}

// RF LQI Thread
DWORD RF_LQIThread(void * pContext)
{
    hprf_hw_dev *pDevice = (hprf_hw_dev*)pContext;
    UINT16 lq_norm_local_device, lq_norm_gateway;
    TYPE_RF_ECHO_REQUEST_PACKET Rf_Serv_Echoreq;
    TYPE_RF_DISCOVER_REQUEST Rf_Serv_adreq;
    HPRF_UNALIGNED UINT8 data[20];

    //////////////////////////////////////////////////////
    // Wait for initialization to complete

    RFIntHostApp.pF_StartupWait();
    while (!RFReceive_IsRFStackInitialized())	{	RFIntHostApp.pF_HwDelayMsec(100);	}
    
    RFIntHostApp.pF_ErrorPrintf("\r\nRF_LQIThread: Running!");

    //////////////////////////////////////////////////////
    // Start the thread
    while (TRUE)
    {
        DWORD dwResult = WaitForSingleObject(pDevice->hLQIEvent, OS_SUSPEND);

        /* Always default the values when the tasks tarts */
        // Note: This is done in the IOCTL handler since this is a manual reset event
        // LocalDevLQI = 255;
        // RemoteLQI = 255;

        //////////////////////////////////////////////////////
        // Actual Event
        if (dwResult == WAIT_OBJECT_0)
        {
            // LQ at the local device (panel) and gateway
            lq_norm_local_device = RFIntRFApp.pF_ConvertLQIToIEEEScaleWord(LocalDevLQI, LQI_DISP_OFFSET);
            lq_norm_gateway = RFIntRFApp.pF_ConvertLQIToIEEEScaleWord(RemoteLQI, LQI_DISP_OFFSET);

            // Build the packet
            data[0] =  HPRF_CTRL_MSG_LQI_DATA;
            *(DWORD*)&data[1] = (DWORD)lq_norm_local_device;
            *(INT*)&data[5]   = (INT)RFIntRFApp.pF_ConvertLQITodBm(LocalDevLQI);
            *(DWORD*)&data[9] = (DWORD)lq_norm_gateway;
            *(INT*)&data[13]  = (INT)RFIntRFApp.pF_ConvertLQITodBm(RemoteLQI);

            // Post Control message to the application
            SendCtrlMsgToApp((void*)data, sizeof (data));

            if (RFIntRFApp.pF_GatewayConnectionStatusQuery() == TRUE)
            {
                memset ((char*)&Rf_Serv_Echoreq, 0, sizeof(Rf_Serv_Echoreq));
                Rf_Serv_Echoreq.rfid    = 0;
                Rf_Serv_Echoreq.repeats = 1;
                Rf_Serv_Echoreq.buflen  = sprintf((char*)Rf_Serv_Echoreq.buf, "TESTLQI"); 
                Rf_Serv_Echoreq.start   = TRUE;         //  TRUE start the test, FALSE stop the test
                Rf_Serv_Echoreq.pcbf    = RF_LQI_Timer_EchoRequest_CB;       
                RFIntRFApp.pF_PostEchoPacketRequest(&Rf_Serv_Echoreq);
            }
            else
            {
                /* Clear the value ( = 255 in dBm) before posting the next message */
                LocalDevLQI = 255;  /* For the echo, it will be cleared automatically */
                RemoteLQI   = 255;
                memset ((char*)&Rf_Serv_adreq, 0, sizeof(Rf_Serv_adreq));
                Rf_Serv_adreq.firstchannel = rfparam.channel;   
                Rf_Serv_adreq.lastchannel  = rfparam.channel;
                Rf_Serv_adreq.service      = RF_PACKET_DISCOVER_NETDEV_REQUEST;
                Rf_Serv_adreq.detialview = FALSE;
                Rf_Serv_adreq.scan  = TRUE;         
                Rf_Serv_adreq.flags = RF_DISCOVER_GATEWAYS;
                Rf_Serv_adreq.waitforto = FALSE;
                Rf_Serv_adreq.pcbf  = RF_LQI_Timer_NetworkService_CB;
                RFIntRFApp.pF_NetworkService (&Rf_Serv_adreq);
            }   

            // Suspend the task for LQI interval
            Sleep(RF_LQI_INTERVAL_MS);
        }
        else
        {
            RETAILMSG(1, (TEXT("RF_LQIThread: Failed Wait Object. Result: %d, Error: %d\r\n"), dwResult, GetLastError()));
        }
    }
    return 0;
}

//-----------------------------------------------------------------------------
//
//  Function: Rf Echo test response call back function
//
//  Parameters:
//          [pechoreply] : Echo reply packet
//
//  Returns:
//      Returns TRUE on success. Returns FALSE on failure.
//
//-----------------------------------------------------------------------------
INT32   RFEchoTestResponse (TYPE_RF_ECHO_REPLY_PACKET *pechoreply)
{
    HPR_TY_DiagData EchoReplyData;
    if (pechoreply->done || ((pechoreply->repeatno % 100) == 0))
    {
        RETAILMSG(1, (TEXT("Echo Test Done!\r\n")));

        // Adjust whatever needes to be, since application won't have these functions.
        (INT8)pechoreply->avgrxlqi = RFReceive_ConvertLinkQualityTodBm(pechoreply->avgrxlqi);
        (INT8)pechoreply->avgremotedevicerxlqi = RFReceive_ConvertLinkQualityTodBm(pechoreply->avgremotedevicerxlqi);
        (INT8)pechoreply->localrxlqi = RFReceive_ConvertLinkQualityTodBm(pechoreply->localrxlqi);
        (INT8)pechoreply->remoterxlqi = RFReceive_ConvertLinkQualityTodBm(pechoreply->remoterxlqi);

        // Fill control code for the control message queue and copy the reply data
        EchoReplyData.dwDataCode = HPRF_DIAG_DATA_ECHO_TEST_REPLY;
        memcpy(&EchoReplyData.Data.EchoReply, pechoreply, sizeof (TYPE_RF_ECHO_REPLY_PACKET));

        // Post Control message to the application
        SendDiagDataToApp((void*)&EchoReplyData, sizeof (HPR_TY_DiagData));
    }
    return SUCCESS;

}
//-----------------------------------------------------------------------------
//
//  Function: Rf Tests
//
//  All RF related tests are here.
//
//  Parameters:
//          [pBufIn]     : Pointer to Control buffer
//          [dwLenIn]    : Length of the in buffer
//          [pBufOut]    : Pointer to out buffer to return the diag information.
//          [dwLenOut]   : Length of the out buffer
//
//  Returns:
//      Returns TRUE on success. Returns FALSE on failure.
//
//-----------------------------------------------------------------------------
DWORD RF_Tests(void* pBufIn, DWORD dwLenIn, void* pBufOut, DWORD dwLenOut)
{
    RF_TEST_CTRL* pTestCtrl;
    RF_TEST_TELEC_DATA* pTelecTestCtrl;
    RF_TEST_ECHO_TEST_DATA* pEchoTestCtrl;
    UINT16 Power;
    UINT8 Channel, Mode;
    DWORD dwResult = FALSE;
    TYPE_RF_ECHO_REQUEST_PACKET echoreq;

    // check if we have right buffer
    if (!pBufIn || dwLenIn < sizeof (RF_TEST_CTRL))
    {
        // Invalid buffer
        RETAILMSG(1, (TEXT("RF_Tests: Invalid input buffer\r\n")));
        return FALSE;
    }

    // First typecast to control structure. All Tests will have its first element
    // as control strcture.
    pTestCtrl = (RF_TEST_CTRL*)pBufIn;

    // Telect Test
    if (pTestCtrl->dwCode == HPR_TEST_CTRL_TELEC_TEST)
    {
        // we must get correct data size
        if (dwLenIn < sizeof (RF_TEST_TELEC_DATA))
        {
            RETAILMSG(1, (TEXT("RF_Tests: Invalid input buffer (telec data)\r\n")));
            return FALSE;
        }
        // Typecast to the actual structure
        pTelecTestCtrl = (RF_TEST_TELEC_DATA*)pBufIn;

        Mode = (UINT8)pTelecTestCtrl->dwMode;
        Channel = (UINT8)pTelecTestCtrl->dwChannel;
        Power = (UINT16)pTelecTestCtrl->dwPower;
        
        DEBUGMSG(ZONE_IOCTL, (TEXT("RF_Tests: [TELEC] Mode[%d], Channel[%d], Power[%d].\r\n"),Mode, Channel, Power));
        
        if ((Mode > gTestContinuousTxNoMod_c) ||
            (Channel < gLogicalChannel11_c || Channel > gLogicalChannel26_c) ||
            (Power > RF_PHY_MAX_POWER_LEVEL))
        {
            DEBUGMSG(ZONE_IOCTL, (TEXT("HPRF.DLL: IOCTL - Telec Test Param error.\r\n")));
        }
        else
        {
            RFTransmit_BuildPowerLevelFrameAndPostToPipe (Power, FALSE);
            RFTransmit_BuildTELCTestChannelMessageAndPostToPipe (Channel);
            RFTransmit_BuildTELCTestMessageAndPostToPipe (Mode);
            dwResult = sizeof (RF_TEST_TELEC_DATA); // indicates success
        }
    }
    else if (pTestCtrl->dwCode == HPR_TEST_CTRL_ECHO_TEST)
    {
        // we must get correct data size
        if (dwLenIn < sizeof (RF_TEST_ECHO_TEST_DATA))
        {
            RETAILMSG(1, (TEXT("RF_Tests: Invalid input buffer (echo data)\r\n")));
            return FALSE;
        }

        // Typecast to the actual structure
        pEchoTestCtrl = (RF_TEST_ECHO_TEST_DATA*)pBufIn;

        // Print debug message
        RETAILMSG(1, (TEXT("RF_Tests: Echo Test\r\n")));

        memset ((UINT8*)&echoreq, 0, sizeof(echoreq));
        echoreq.rfid    = (UINT8)pEchoTestCtrl->dwRFId;
        echoreq.repeats = pEchoTestCtrl->dwRepeat;
        memcpy (echoreq.buf, pEchoTestCtrl->bDataBuffer, pEchoTestCtrl->dwDataLen);
        echoreq.buflen  = pEchoTestCtrl->dwDataLen;  
        echoreq.start   = TRUE;
        echoreq.pcbf    = RFEchoTestResponse;
        RFIntRFApp.pF_PostEchoPacketRequest(&echoreq);

        dwResult = sizeof (RF_TEST_ECHO_TEST_DATA);

    }
    else if (pTestCtrl->dwCode == HPR_TEST_CTRL_STACK_RESET)
    {
        // Print debug message
        RETAILMSG(1, (TEXT("RF_Tests: Stack Reset\r\n")));
        RFTransmit_BuildResetMACMessageAndPostToPipe (1);
        dwResult = sizeof (RF_TEST_CTRL);
    }
    return (dwResult);
}

/*-----------------12/3/01 10:04AM------------------
 * Function:        ConvertTimeToHMS
 *
 * Returns:         char *
 *
 * Return Values:   pointer to destination string
 *
 * Description:     This routine converts the time stamp to days hh:mm:ss.ss format
 *
 * Parameters:      unsigned long Timestamp - number of ticks since bootup in 1/100 s
 *                  char *DestString - pointer to destination
 * Reentrant
 * --------------------------------------------------*/
#define TNUM_DAYS     (24*60*60*1000)
#define TNUM_HOURS    (60*60*1000)
#define TNUM_MINS     (60*1000)
#define TNUM_SECS     (1000)

char *ConvertTimeToHMS(unsigned long Timestamp, char *DestString)
{
    int Days, Hours, Mins, Secs;

    Days = Timestamp / TNUM_DAYS;
    Timestamp = Timestamp % TNUM_DAYS;
    Hours = Timestamp / TNUM_HOURS;
    Timestamp = Timestamp % TNUM_HOURS;
    Mins = Timestamp / TNUM_MINS;
    Timestamp = Timestamp % TNUM_MINS;
    Secs = Timestamp / TNUM_SECS;
    Timestamp = Timestamp % TNUM_SECS;

    sprintf(DestString, "%d days %02d:%02d:%02d.%02d", Days, Hours, Mins, Secs, Timestamp);
    return(DestString);
}

//-----------------------------------------------------------------------------
//
//  Function: Rf diagnostics
//
//  All RF diagnostics related are here.
//
//  Parameters:
//          [pBufIn]     : Pointer to Control buffer
//          [dwLenIn]    : Length of the in buffer
//          [pBufOut]    : Pointer to out buffer to return the diag information.
//          [dwLenOut]   : Length of the out buffer
//
//  Returns:
//      Returns TRUE on success. Returns FALSE on failure.
//
//-----------------------------------------------------------------------------
DWORD RF_Diagnostics(void* pBufIn, DWORD dwLenIn, void* pBufOut, DWORD dwLenOut)
{
    RF_DIAG_CTRL *pDiagCtrl;
    RF_DIAG_SHOW_DIAG_INFO* pShowDiagInfo;
    RF_PHY_INFO* pPhyInfo;

    UINT8   IndexArray[MAX_DESCRIPTIVE_TABLE_ENTRIES];
    UINT16 count;
    TYPE_RF_DESCRIPTIVE_TABLE *pDesct;
    UINT8 *pindex;

    RF_DIAG_DESCR* pDiagDescr;
    DWORD dwTemp = 0;
    FLOAT fTemp = 0;
    HPR_TY_DiagData DescrData;
    INT8    buf[100];
    UNSIGNED pipesize, avaliable, messages;

    TYPE_RF_STACK_TRACE* pStackTrace;

    TYPE_RF_STACK_DIAG* pStackDiag;
    UINT8 msgindex, msgentries, errpktcnt;

    TYPE_RF_PHY_READ *pPhyReadIn, *pPhyReadOut;
    UINT8 addr;

#ifdef HPRF_ERR_MSG_LOG
    HPR_TY_DiagData Diag;
    TYPE_RF_DIAG_ERR_LOG* pErrLog;
    CHAR* pStrPtr;
#endif

    TYPE_RF_NETLINK_INFO* pRFNetlinkInfo;

    TYPE_RF_VERSION_INFO* pRfVersion;
    TYPE_RF_POWER_DBM* pRfPowerDbm;

    // check if we have right buffer
    if (!pBufIn || dwLenIn < sizeof (RF_DIAG_CTRL))
    {
        // Invalid buffer
        RETAILMSG(1, (TEXT("RF_Diagnostics: Invalid input buffer\r\n")));
        return FALSE;
    }

    // Typecast the strcture.
    pDiagCtrl = (RF_DIAG_CTRL*)pBufIn;

    // For now, later can be converted to a switch statement.
    if (pDiagCtrl->dwCode == HPR_DIAG_CTRL_SHOW_DIAG_INFO)
    {
        // Check the outbuffer
        if (!pBufOut || dwLenOut < sizeof (RF_DIAG_SHOW_DIAG_INFO))
        {
            RETAILMSG(1, (TEXT("RF_Diagnostics: Invalid output buffer\r\n")));
            return FALSE;
        }

        // Typecast output buffer
        pShowDiagInfo = (RF_DIAG_SHOW_DIAG_INFO*)pBufOut;

        count = RFDesc_GetIndexToDescriptiveTableAllDevices (IndexArray);
        pindex= IndexArray;
        while (count-- && (*pindex < g_MaxSupportedRFDevices))
        {
            pDesct = RFDesc_GetDescriptiveEntryIDInfo(*pindex);
            *pindex++;
        }

        // Now assign values to strcutre members
        pShowDiagInfo->linkstatecnt = pDesct->linkstatecnt;
        pShowDiagInfo->linkqualityAtLocalDevice = pDesct->LQ.linkqualityAtLocalDevice;
        pShowDiagInfo->txpacketcount = pDesct->COMLog.txpacketcount;
        pShowDiagInfo->txpacketretrycount = pDesct->COMLog.txpacketretrycount;
        pShowDiagInfo->rxpacketcount = pDesct->COMLog.rxpacketcount;
        pShowDiagInfo->rxpacketrepeatcount = pDesct->COMLog.rxpacketrepeatcount;
        pShowDiagInfo->rxpacketsequenceErrcount = pDesct->COMLog.rxpacketsequenceErrcount;
        pShowDiagInfo->rxpacketsegmentationerrcnt = pDesct->COMLog.rxpacketsegmentationerrcnt;

        // Low level diagnostics (as that in TPS-6X setup screen diag display, sub page)
        pShowDiagInfo->rfdiag.Rfdiags_numTimer1Ints = Rfdiags_numTimer1Ints;
        pShowDiagInfo->rfdiag.Rfdiags_numLOUnLockInts = Rfdiags_numLOUnLockInts;
        pShowDiagInfo->rfdiag.Rfdiags_numInvalidPhyPacketsRecvd = Rfdiags_numInvalidPhyPacketsRecvd;
        pShowDiagInfo->rfdiag.Rfdiags_NumCCAFailures = Rfdiags_NumCCAFailures;
        pShowDiagInfo->rfdiag.Rfdiags_LowLevelRecvCnt = Rfdiags_LowLevelRecvCnt;
        pShowDiagInfo->rfdiag.Rfdiags_LowLevelRecvByteCnt = Rfdiags_LowLevelRecvByteCnt;
        pShowDiagInfo->rfdiag.Rfdiags_LowLevelTotalLostPackets = Rfdiags_LowLevelTotalLostPackets;
        pShowDiagInfo->rfdiag.Rfdiags_LowLevelInvalidRxPacketCnt = Rfdiags_LowLevelInvalidRxPacketCnt;
        pShowDiagInfo->rfdiag.Rfdiags_LowLevelRetryPacketCnt = Rfdiags_LowLevelRetryPacketCnt;
        pShowDiagInfo->rfdiag.Rfdiags_LowLevelTxPacketCnt = Rfdiags_LowLevelTxPacketCnt;
        pShowDiagInfo->rfdiag.Rfdiags_LowLevelTxPacketBytes = Rfdiags_LowLevelTxPacketBytes;

        // Following are not present in TPS-6X setup screen display
        pShowDiagInfo->rfdiag.Rfdiags_numCCA_TxDone = Rfdiags_numCCA_TxDone;
        pShowDiagInfo->rfdiag.Rfdiags_LowLevelResets = Rfdiags_LowLevelResets;
        pShowDiagInfo->rfdiag.Rfdiags_numBadTimerISR = Rfdiags_numBadTimerISR;
        pShowDiagInfo->rfdiag.Rfdiags_ChannelBusyResets = Rfdiags_ChannelBusyResets;
        pShowDiagInfo->rfdiag.Rfdiags_NumDataConfMessages = Rfdiags_NumDataConfMessages;
        pShowDiagInfo->rfdiag.Rfdiags_NumResetAndNacks = Rfdiags_NumResetAndNacks;
        pShowDiagInfo->rfdiag.Rfdiags_Congestion = Rfdiags_Congestion;
        pShowDiagInfo->rfdiag.Rfdiags_Utilization = Rfdiags_Utilization;
        pShowDiagInfo->rfdiag.Rfdiags_NumCCATimeoutsFailures = Rfdiags_NumCCATimeoutsFailures;
        pShowDiagInfo->rfdiag.Rfdiags_NumRxInactivityResets = Rfdiags_NumRxInactivityResets;

        // PHY Information
        pShowDiagInfo->PhyInfo.PhyInitFailed = Rfdiags_LowLevelPhyInitFailures;
        pShowDiagInfo->PhyInfo.PhyChipId = Rfdiags_LowLevelPhyChipId;

        // On success, return size of the structure
        return (sizeof (RF_DIAG_SHOW_DIAG_INFO));
    }
    if (pDiagCtrl->dwCode == HPR_DIAG_CTRL_PHY_INFO)
    {
        // Check the outbuffer
        if (!pBufOut || dwLenOut < sizeof (RF_PHY_INFO))
        {
            RETAILMSG(1, (TEXT("RF_Diagnostics: Invalid output buffer\r\n")));
            return FALSE;
        }

        // Typecast output buffer
        pPhyInfo = (RF_PHY_INFO*)pBufOut;
        
        // Fill buffer
        pPhyInfo->PhyInitFailed = Rfdiags_LowLevelPhyInitFailures;
        pPhyInfo->PhyChipId = Rfdiags_LowLevelPhyChipId;

        return (sizeof (RF_PHY_INFO));
    }
    if (pDiagCtrl->dwCode == HPR_DIAG_CTRL_DESCR)
    {
        // Type cast the input buffer
        pDiagDescr = (RF_DIAG_DESCR*)pBufIn;
        
        // Read the max descriptive table entry.
        dwTemp = RFDesc_MaxSupportedDescriptiveEntries();
        //RETAILMSG(1, (TEXT("RF_Diag: Descriptor (Index: %d, Max supported: %d)\r\n"), pDiagDescr->index, dwTemp));
        
        // Only if the index is less than the max supported index
        if (pDiagDescr->index < dwTemp)
        {
            pDesct = RFDesc_GetDescriptiveEntryIDInfo((UINT8)pDiagDescr->index);
            if (pDesct == NULL)
            {
                RETAILMSG(1, (TEXT("RF_Diagnostics: RF Descriptor NULL\r\n")));
                // Type cast the output buffer
                pDiagDescr = (RF_DIAG_DESCR*)pBufOut;
                pDiagDescr->index = dwTemp;
                return FALSE;
            }
            // Fill in data and control code and send to app.
            DescrData.dwDataCode = HPRF_DIAG_DATA_DESCR_COMMAND_REPLY;

            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " [RF DESCRIPTOR REPLY]\r\n");
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));

            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Descriptive entry:                     %d\r\n", pDiagDescr->index);
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " IDstring:                              %s\r\n",   pDesct->IDInfo.idstring);
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " UID:                                   %s\r\n",   ConvertUIDtoString ((UINT8*)&pDesct->IDInfo.uid, buf, gAddrModeLong_c));
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Panid:                                 %04x\r\n", pDesct->IDInfo.panid);
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Addrmode:                              %d\r\n", pDesct->IDInfo.addrmode);
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " RFID:                                  %02x\r\n", pDesct->IDInfo.rfid);
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " TSID:                                  %08x\r\n", pDesct->IDInfo.tsid);
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Power:                                 %d\r\n", pDesct->IDInfo.power);
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Deviceoptions:                         %d\r\n", pDesct->IDInfo.deviceoptions);
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Link state:                            %s\r\n",   ConvertLinkStatetoString (pDesct->linkstate, buf));  
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Link state connection count:           %u\r\n", pDesct->linkstatecnt);
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            
            //  new
            ConvertTimeToHMS((OS_RetrieveClock() - pDesct->linkeactiveestablishedtime), buf);
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Link active time:                      %u (%s)\r\n", pDesct->linkeactiveestablishedtime, buf);
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));

            //  new
            RFTransmit_GetPipeAvaliableMemorySizeByIndex ((UINT8)pDiagDescr->index, &pipesize, &avaliable, &messages);
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Transmit Pipe info size/avaliable/msgs:%u/%u/%u\r\n", pipesize,avaliable,messages);  
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Linkquality at local device:           %d (%d dBm)\r\n", pDesct->LQ.linkqualityAtLocalDevice, RFReceive_ConvertLinkQualityTodBm(pDesct->LQ.linkqualityAtLocalDevice));
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Linkquality at remote device:          %d (%d dBm)\r\n", pDesct->LQ.linkqualityAtRemoteDevice, RFReceive_ConvertLinkQualityTodBm(pDesct->LQ.linkqualityAtRemoteDevice));
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Waiting for ack flag:                  %d\r\n",   pDesct->waitingforACK);
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Result code available flag:            %d\r\n",   pDesct->resultcodeavailable);
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Result code:                           %d\r\n", pDesct->resultcode);
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Retry timer:                           %u\r\n", pDesct->RetryTimer);
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Retry timed out:                       %d\r\n", pDesct->RetryTimedOut);
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Tx packet retry cntr:                  %d\r\n", pDesct->txpacketretrycntr);
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Maximum Tx packet retries:             %d\r\n", pDesct->maxtxpacketretries);
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Heart beat timer:                      %d\r\n", pDesct->HeartBeatTimer);
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Heart beat timed out:                  %d\r\n",   pDesct->HeartBeatTimedOut);
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Result code timer:                     %d\r\n", pDesct->ResultCodeTimer);
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Result code timed out:                 %d\r\n",   pDesct->ResultCodeTimedOut);
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Tx packet sequence number:             %d\r\n", pDesct->txsequence);
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of tx packets:                %u\r\n", pDesct->COMLog.txpacketcount);           
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of tx packet retries:         %u\r\n", pDesct->COMLog.txpacketretrycount);          
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of tx packet in errors:       %u\r\n", pDesct->COMLog.txpacketerrorcount);          
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of times tx buffer overflow:  %u\r\n", pDesct->COMLog.txbufoverflowcount);          
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total tx packet resultcode timeouts:   %u\r\n", pDesct->COMLog.txpacketresultcodetimeouts);
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));

            // new
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of concatenated tx packet:    %u\r\n", pDesct->COMLog.txpacketconcatcount);
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of segmented tx packet:       %u\r\n", pDesct->COMLog.txpacketsegmented);
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));

            //printf(" tx packet                              %u\r\n", pDesct->COMLog.txpacketfailcount);         
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total tx packet payload data in bytes: %u\r\n", pDesct->COMLog.txpacketpayloadbytecount);            
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total tx packet overhead in bytes:     %u\r\n", pDesct->COMLog.txpacketoverheadbytecount);           
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
        
            //  modified
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Average tx packet length in bytes:     %u\r\n", pDesct->COMLog.txpacketcount?
                                                        ((pDesct->COMLog.txpacketoverheadbytecount+pDesct->COMLog.txpacketpayloadbytecount)/pDesct->COMLog.txpacketcount):0);           
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));

            //  new
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Tx packet turn around delay in 10msec: %u\r\n",    pDesct->COMLog.txpacketturnaroundtime);
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Tx packet avg turn around delay in 10msec: %u\r\n",    (pDesct->COMLog.txpacketcount-pDesct->COMLog.txpacketretrycount)?(pDesct->COMLog.txpacketaccturnaroundtime/(pDesct->COMLog.txpacketcount-pDesct->COMLog.txpacketretrycount)):0);
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));

            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Rx packet sequence number:             %d\r\n",  pDesct->rxsequence);            
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of rx packets:                %u\r\n",  pDesct->COMLog.rxpacketcount);          
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of rx packet repeats:         %u\r\n",  pDesct->COMLog.rxpacketrepeatcount);            
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of rx packet sequence err:    %u\r\n",  pDesct->COMLog.rxpacketsequenceErrcount);           
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of rx packet segment err:     %u\r\n",  pDesct->COMLog.rxpacketsegmentationerrcnt);
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of rx packet with invalid len:%u\r\n",  pDesct->COMLog.rxpacketlenerror);
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of rx packet with hdr/footr err:%u\r\n",    pDesct->COMLog.rxpackethdrftrerror);
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of times gw rx buffer overflow: %u\r\n",  pDesct->COMLog.rxbufoverflowcount);  
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
    
            // new
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of rx packet concatenation:   %u\r\n", pDesct->COMLog.rxpacketconcatcount);
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of rx acks:                   %u\r\n", pDesct->COMLog.rxpacketacks);          
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of rx naks/cca's:             %u\r\n", pDesct->COMLog.rxpacketnaks);          
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total rx packet payload data in bytes: %u\r\n",    pDesct->COMLog.rxpacketpayloadbytecount);       
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total rx gw startup packets:           %u\r\n",    pDesct->COMLog.rxgwrestarts);       
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no of heartbeat timeouts:        %u\r\n",    pDesct->COMLog.heartbeattimeouts);          
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            DescrData.Data.DescReply.dwStrLen = sprintf(DescrData.Data.DescReply.Buffer, " Total no tx cong. avoidance:           %u\r\n", pDesct->COMLog.txpacketcongavoidance);
            SendDiagDataToApp((void*)&DescrData, sizeof (HPR_TY_DiagData));
            
            return (sizeof (RF_DIAG_DESCR));
        }
        else
        {
            RETAILMSG(1, (TEXT("RF_Diagnostics: RF Descriptor out of bound\r\n")));
            // Type cast the output buffer
            pDiagDescr = (RF_DIAG_DESCR*)pBufOut;
            pDiagDescr->index = dwTemp;
            return FALSE;

        }
    }
    if (pDiagCtrl->dwCode == HPR_DIAG_CTRL_RF_STACK_TRACE)
    {
        RETAILMSG(1, (TEXT("RF_Diag: Stack trace.\r\n")));
        
        // Check buffer validity before typecasting...
        if (!pBufOut || dwLenOut < sizeof (TYPE_RF_STACK_TRACE))
        {
            RETAILMSG(1, (TEXT("RF_Diag: Invalid stack trace buffer.\r\n")));
            return FALSE;
        }

        // Out buffer is a pointer to the stack trace structure.
        pStackTrace = (TYPE_RF_STACK_TRACE*)pBufOut;
        // Fill the buffer
        pStackTrace->SMAC_SM_Debug_State_Ctr = SMAC_SM_Debug_State_Ctr;
        for (dwTemp = 0; dwTemp < SMAC_SM_DEBUG_NUMENTRIES; dwTemp++)
        {
            pStackTrace->SMAC_SM_Debug_State[dwTemp] = SMAC_SM_Debug_State[dwTemp];
            pStackTrace->SMAC_SM_Debug_Timestamp[dwTemp] = SMAC_SM_Debug_Timestamp[dwTemp];
        }
        return (sizeof (TYPE_RF_STACK_TRACE));
    }
    if (pDiagCtrl->dwCode == HPR_DIAG_CTRL_RF_CLR_DIAG)
    {
        RETAILMSG(1, (TEXT("RF_Diag: Clear RF diag counters.\r\n")));
        //  Clear diagnostic counters
        Rfdiags_numTimer1Ints   = 0;
        Rfdiags_numLOUnLockInts = 0;
        Rfdiags_numInvalidPhyPacketsRecvd = 0;
        Rfdiags_numCCA_TxDone = 0;
        Rfdiags_LowLevelRecvCnt = 0;
        Rfdiags_LowLevelRecvByteCnt = 0;
        Rfdiags_LowLevelTotalLostPackets = 0;
        Rfdiags_LowLevelInvalidRxPacketCnt = 0;
        Rfdiags_LowLevelRetryPacketCnt = 0;
        Rfdiags_LowLevelTxPacketCnt = 0;
        Rfdiags_LowLevelTxPacketBytes = 0;          
        Rfdiags_NumCCAFailures = 0;
        Rfdiags_NumRxInactivityResets = 0;
        return (sizeof (RF_DIAG_CTRL));
    }
    if (pDiagCtrl->dwCode == HPR_DIAG_CTRL_RF_STACK_DIAG)
    {
        RETAILMSG(1, (TEXT("RF_Diag: Stack diag.\r\n")));
        if (!pBufOut || dwLenOut < sizeof (TYPE_RF_STACK_DIAG))
        {
            RETAILMSG(1, (TEXT("RF_Diag: Invalid stack diag buffer\r\n")));
            return FALSE;
        }

        // Now typecast
        pStackDiag = (TYPE_RF_STACK_DIAG*)pBufOut;

        // Fill the data
        pStackDiag->rfdiag.Rfdiags_numTimer1Ints = Rfdiags_numTimer1Ints;
        pStackDiag->rfdiag.Rfdiags_numLOUnLockInts = Rfdiags_numLOUnLockInts;
        pStackDiag->rfdiag.Rfdiags_numInvalidPhyPacketsRecvd = Rfdiags_numInvalidPhyPacketsRecvd;
        pStackDiag->rfdiag.Rfdiags_numCCA_TxDone = Rfdiags_numCCA_TxDone;
        pStackDiag->rfdiag.Rfdiags_LowLevelRecvCnt = Rfdiags_LowLevelRecvCnt;
        pStackDiag->rfdiag.Rfdiags_LowLevelRecvByteCnt = Rfdiags_LowLevelRecvByteCnt;
        pStackDiag->rfdiag.Rfdiags_LowLevelTotalLostPackets = Rfdiags_LowLevelTotalLostPackets;
        pStackDiag->rfdiag.Rfdiags_LowLevelInvalidRxPacketCnt = Rfdiags_LowLevelInvalidRxPacketCnt;
        pStackDiag->rfdiag.Rfdiags_LowLevelRetryPacketCnt = Rfdiags_LowLevelRetryPacketCnt;
        pStackDiag->rfdiag.Rfdiags_LowLevelTxPacketCnt = Rfdiags_LowLevelTxPacketCnt;
        pStackDiag->rfdiag.Rfdiags_LowLevelTxPacketBytes = Rfdiags_LowLevelTxPacketBytes;
        pStackDiag->rfdiag.Rfdiags_LowLevelResets = Rfdiags_LowLevelResets;
        pStackDiag->rfdiag.Rfdiags_numBadTimerISR = Rfdiags_numBadTimerISR;
        pStackDiag->rfdiag.Rfdiags_ChannelBusyResets = Rfdiags_ChannelBusyResets;
        pStackDiag->rfdiag.Rfdiags_NumDataConfMessages = Rfdiags_NumDataConfMessages;
        pStackDiag->rfdiag.Rfdiags_NumResetAndNacks = Rfdiags_NumResetAndNacks;
        pStackDiag->rfdiag.Rfdiags_Congestion = Rfdiags_Congestion;
        pStackDiag->rfdiag.Rfdiags_Utilization = Rfdiags_Utilization;
        pStackDiag->rfdiag.Rfdiags_NumCCAFailures = Rfdiags_NumCCAFailures;
        pStackDiag->rfdiag.Rfdiags_NumCCATimeoutsFailures = Rfdiags_NumCCATimeoutsFailures;
        pStackDiag->rfdiag.Rfdiags_NumRxInactivityResets = Rfdiags_NumRxInactivityResets;

        for (dwTemp = 0; dwTemp < g_MaxSupportedRFDevices; dwTemp++)
            pStackDiag->numResCodeTimeouts += g_pRFDeviceDescriptiveTable[dwTemp].COMLog.txpacketresultcodetimeouts;
        
        RFDesc_GetErrorPktIndex (DEVICE_DESCRIPTIVE_TABLE_INDEX, &msgindex, &msgentries, &errpktcnt);
        pStackDiag->errpktcnt = (DWORD)errpktcnt;

        return (sizeof (TYPE_RF_STACK_DIAG));
    }
    if (pDiagCtrl->dwCode == HPR_DIAG_CTRL_RF_PHY_READ)
    {
        if (!pBufOut || dwLenIn < sizeof (TYPE_RF_PHY_READ)|| dwLenOut < sizeof (TYPE_RF_PHY_READ))
        {
            RETAILMSG(1, (TEXT("RF_Diag: Invalid RF PHY In-Out buffers.\r\n")));
            return FALSE;
        }
        // Type cast in and out buffers (should be of the same type.
        pPhyReadIn = (TYPE_RF_PHY_READ*)pBufIn;
        pPhyReadOut = (TYPE_RF_PHY_READ*)pBufOut;

        // Start at the Read address....
        dwTemp = 0;
        addr = (UINT8)pPhyReadIn->ReadAddr;
        count = (UINT16)pPhyReadIn->ReadCount;
        //RETAILMSG(1, (TEXT("RF_Diag: Read PHY (Reg: 0x%02x, Count: %d)\r\n"),addr, count));

        while (count && addr < MC1319xDRV_MAX_REGS)
        {
            // Now read and fill the buffer
            pPhyReadOut->Data[dwTemp] = (DWORD) SPIDrvRead(addr);
            count--;
            addr++;
            dwTemp++;
        }
        return (sizeof (TYPE_RF_PHY_READ));
    }
#ifdef HPRF_ERR_MSG_LOG
    if (pDiagCtrl->dwCode == HPR_DIAG_CTRL_RF_ERR_LOG)
    {
        if (dwLenIn < sizeof (TYPE_RF_DIAG_ERR_LOG))
        {
            RETAILMSG(1, (TEXT("RF_Diag: Incorrect ERRORLOG input buffer.\r\n")));
            return FALSE;
        }
        RETAILMSG(1, (TEXT("RF_Diag: RF Error Log (Msgs: %u, Total bytes: %u.\r\n"),
                    RfErrorLog.NoErrorMsgs, RfErrorLog.TotalBytes));
        
        // Typecast.
        pErrLog = (TYPE_RF_DIAG_ERR_LOG*)pBufIn;
        
        // Fill the control code...
        Diag.dwDataCode = HPRF_DIAG_DATA_CONSOLE_PRINT;
        // If we have atleast one error messages logged.....
        if (RfErrorLog.NoErrorMsgs)
        {
            // Start from beginning...
            pStrPtr = RfErrorLog.RFErrorMsgBuff;
            // Send all messages to application...
            for (dwTemp = 0; dwTemp < RfErrorLog.NoErrorMsgs; dwTemp++)
            {
                strcpy(Diag.Data.Console.buffer, pStrPtr);
                Diag.Data.Console.dwStrLen = strlen(pStrPtr);
                
                // Send the string to application...
                SendDiagDataToApp((void*)&Diag, sizeof (HPR_TY_DiagData));

                // Advance the pointer to next string... (consider NULL)
                pStrPtr = pStrPtr + Diag.Data.Console.dwStrLen + 1;
            }
            // clear command...
            if (pErrLog->Control == 1)
            {
                // clear the error log...
                RfErrorLog.NoErrorMsgs = 0;
                RfErrorLog.TotalBytes = 0;
                // Send clear message...
                strcpy(Diag.Data.Console.buffer, "\r\nRF Error logs cleared!");
                Diag.Data.Console.dwStrLen = strlen(Diag.Data.Console.buffer);
                SendDiagDataToApp((void*)&Diag, sizeof (HPR_TY_DiagData));
            }
        }
        else
        {
            strcpy(Diag.Data.Console.buffer, "\r\nNo RF Error Messages logged!");
            Diag.Data.Console.dwStrLen = strlen(Diag.Data.Console.buffer);
            SendDiagDataToApp((void*)&Diag, sizeof (HPR_TY_DiagData));
        }

        return (sizeof (TYPE_RF_DIAG_ERR_LOG));
    }
#endif
    if (pDiagCtrl->dwCode == HPR_DIAG_CTRL_RF_NETLINK)
    {
        // Check buffer size...
        if (!pBufOut || dwLenOut < sizeof (TYPE_RF_NETLINK_INFO))
        {
            RETAILMSG(1, (TEXT("RF_Diag: Invalid RF NETLINK Out buffer.\r\n")));
            return FALSE;
        }
        RETAILMSG(1, (TEXT("RF_Diag: RF Netlink.\r\n")));

        // Type cast output buffer
        pRFNetlinkInfo = (TYPE_RF_NETLINK_INFO*)pBufOut;

        // Fill the info..
        pRFNetlinkInfo->GwConn = RFIntRFApp.pF_GatewayConnectionStatusQuery();
        pRFNetlinkInfo->CtrlSysConn = RFIntRFApp.pF_ControlSystemConnectionStatusQuery();
        if (RFReceive_IsControlSystemConnectedNetTypeCresnet())
        {
            pRFNetlinkInfo->GwCtrlConnType = 1;
        }
        else if (RFReceive_IsControlSystemConnectedNetTypeEthernet())
        {
            pRFNetlinkInfo->GwCtrlConnType = 2;
        }
        else
        {
            pRFNetlinkInfo->GwCtrlConnType = 0;
        }

        IndexArray[0] = DEVICE_DESCRIPTIVE_TABLE_INDEX;
        pindex = IndexArray;
        count = 1;

        RFDesc_SortIndexElementsByRFID (pindex, (UINT8)count);
        pDesct = RFDesc_GetDescriptiveEntryIDInfo(*pindex);
        if (pDesct)
        {
            ConvertLinkStatetoString (g_pRFDeviceDescriptiveTable[*pindex].linkstate, pRFNetlinkInfo->LinkStateStr);
            pRFNetlinkInfo->TimeStr[0] = 0;
            if (pDesct->linkstate == RF_DEVICE_LINK_STATE_ACTIVE)
            {
                ConvertTimeToHMS((OS_RetrieveClock() - pDesct->linkeactiveestablishedtime), pRFNetlinkInfo->TimeStr);
            }
            pRFNetlinkInfo->GwCnetId = g_pRFDeviceDescriptiveTable[*pindex].IDInfo.rfid;
            pRFNetlinkInfo->LinkCount = pDesct->linkstatecnt;
            pRFNetlinkInfo->RxLQIinDev = RFReceive_ConvertLinkQualityToPercent(pDesct->LQ.linkqualityAtLocalDevice, LQI_DISP_OFFSET);
            pRFNetlinkInfo->RxLQIinGW = RFReceive_ConvertLinkQualityToPercent(pDesct->LQ.linkqualityAtRemoteDevice, LQI_DISP_OFFSET);
        }
        return (sizeof (TYPE_RF_NETLINK_INFO));
    }
    if (pDiagCtrl->dwCode == HPR_DIAG_CTRL_RF_VERSION)
    {
        if (!pBufOut || dwLenOut < sizeof (TYPE_RF_VERSION_INFO))
        {
            RETAILMSG(1, (TEXT("RF_Diag: Invalid RF Version Out buffer.\r\n")));
            return FALSE;
        }
        RETAILMSG(1, (TEXT("RF_Diag: RF Version.\r\n")));

        // Type cast...
        pRfVersion = (TYPE_RF_VERSION_INFO*)pBufOut;
        strcpy(pRfVersion->RFVersion, RF_Version);
        strcpy(pRfVersion->SmacVersion, SMAC_Version);

        return (sizeof (TYPE_RF_VERSION_INFO));
    }
    if (pDiagCtrl->dwCode == HPR_DIAG_CTRL_RF_POWER_DBM)
    {
        if (!pBufOut || dwLenIn < sizeof (TYPE_RF_POWER_DBM) || dwLenOut < sizeof (TYPE_RF_POWER_DBM))
        {
            RETAILMSG(1, (TEXT("RF_Diag: Invalid RF Power level dBm in or out buffer.\r\n")));
            return FALSE;
        }
        // Typecast
        pRfPowerDbm = (TYPE_RF_POWER_DBM*)pBufIn;
        fTemp = RFIntRFApp.pF_ConvertPowerTodBm((UINT16)pRfPowerDbm->Power, (BOOL)pRfPowerDbm->Translate);
        
        // Send the output back
        pRfPowerDbm = (TYPE_RF_POWER_DBM*)pBufOut;
        pRfPowerDbm->Power_dBm = fTemp;
        
        RETAILMSG(1, (TEXT("RF_Diag: RF Power dBm: %2.2f\r\n"), pRfPowerDbm->Power_dBm));

        return (sizeof (TYPE_RF_POWER_DBM));
    }

    // Unknown, if it reaches here.
    RETAILMSG(1, (TEXT("RF_Diagnostics: Invalid control code: %d\r\n"), pDiagCtrl->dwCode));
    return FALSE;
}


////////////////////////////////////////////////////////////////////
///// EEPROM management start
////////////////////////////////////////////////////////////////////
//-----------------------------------------------------------------------------
//
//  Function: HPRF_ReadWriteEEPROM
//
//  This function reads or writes EEPROM section data what is stored
//  in the flash. To update a param in the EEPROM, first use this
//  function to read the param in to a local structure, and then
//  modify the required param and write it back.
//
//  Parameters:
//          [pEepParam]     : Pointer to EEPROM_PARAMS structure
//          [bReadWrite]    : Read Write flag (1 -> Read, 0 -> Write)
//
//  Returns:
//      Returns TRUE on success. Returns FALSE on failure.
//
//-----------------------------------------------------------------------------
BOOL HPRF_ReadWriteEEPROM(EEP_PARAMS* pEepParam, BOOL bReadWrite)
{
	HANDLE hDriver;
    DWORD dwRet;
    BOOL bRetVal = FALSE;
    DWORD dwTemp;
	long dsOffset;
	DWORD dwSize;


	hDriver  = CreateFile(TEXT("PDS1:"),GENERIC_READ|GENERIC_WRITE,FILE_SHARE_READ|FILE_SHARE_WRITE,NULL,OPEN_EXISTING,0,NULL);
    if (hDriver == INVALID_HANDLE_VALUE)
    {
        RETAILMSG(1, (TEXT("\r\nERROR HPRF_ReadWriteEEPROM: - Failed to open PDS1 handle.\r\n")));
        goto CleanUp;
	}

	if(!(SetFilePointer(hDriver,0,NULL,FILE_END) >= SYSTEM_STORAGE_SIZE))
		RETAILMSG(1, (TEXT("\r\nERROR HPRF_ReadWriteEEPROM: - Invalid Size.\r\n")));

    // Read
    if (bReadWrite)
    {
		SetFilePointer(hDriver,CS_UC_ER_UID,NULL,FILE_BEGIN);
        ReadFile(hDriver,pEepParam->RfUID,8,&dwRet,NULL);
		if (dwRet != UID_MAX_SIZE)
        {
            RETAILMSG(1, (TEXT("\r\nERROR HPRF_ReadWriteEEPROM: Read fail CS_UC_ER_UID dwRet = 0x%x\r\n"),dwRet));
            goto CleanUp;
        }

		SetFilePointer(hDriver,CS_UL_SIGNATURE,NULL,FILE_BEGIN);
		ReadFile(hDriver,&pEepParam->dwSignature,4,&dwRet,NULL);
        if (dwRet != sizeof(DWORD))
        {
            RETAILMSG(1, (TEXT("\r\nERROR HPRF_ReadWriteEEPROM: Read fail CS_UL_SIGNATURE dwRet = 0x%x\r\n"),dwRet));
            goto CleanUp;
        }

		SetFilePointer(hDriver,CS_UL_REVISIONID,NULL,FILE_BEGIN);
		ReadFile(hDriver,&pEepParam->dwVersion,4,&dwRet,NULL);
		if (dwRet != sizeof(DWORD))
        {
            RETAILMSG(1, (TEXT("\r\nERROR HPRF_ReadWriteEEPROM: Read fail CS_UL_REVISIONID dwRet = 0x%x\r\n"),dwRet));
            goto CleanUp;
        }

		SetFilePointer(hDriver,CS_UL_ER_UIDSIGN,NULL,FILE_BEGIN);
		ReadFile(hDriver,&dwTemp,4,&dwRet,NULL);
		if (dwRet != sizeof(DWORD))
        {
            RETAILMSG(1, (TEXT("\r\nERROR HPRF_ReadWriteEEPROM: Read fail CS_UL_ER_UIDSIGN dwRet = 0x%x\r\n"),dwRet));
            goto CleanUp;
        }
		pEepParam->RfUIDSign = dwTemp;
        // See if read is success

		SetFilePointer(hDriver,CS_US_ER_POWER_TABLE,NULL,FILE_BEGIN);
        ReadFile(hDriver,&pEepParam->RfPowerTable,sizeof(pEepParam->RfPowerTable),&dwRet,NULL);
		if (dwRet != sizeof(UINT16))
        {
            RETAILMSG(1, (TEXT("\r\nERROR HPRF_ReadWriteEEPROM: Read fail CS_US_ER_POWER_TABLE dwRet = 0x%x\r\n"),dwRet));
            goto CleanUp;
        }
    }
    else
    {
        // Write it
		// Only allow writing of UID, not signature or revision
		SetFilePointer(hDriver,CS_UC_ER_UID,NULL,FILE_BEGIN);
    	WriteFile(hDriver,pEepParam->RfUID,8,&dwRet,NULL);
        // Write success is indicated by returning length of the structure.
        if (dwRet != UID_MAX_SIZE)
        {
            RETAILMSG(1, (TEXT("\r\nERROR HPRF_ReadWriteEEPROM: Write fail.\r\n")));
            goto CleanUp;
        }

		dwTemp = UID_SIGN;
		SetFilePointer(hDriver,CS_UL_ER_UIDSIGN,NULL,FILE_BEGIN);
    	WriteFile(hDriver,&dwTemp,4,&dwRet,NULL);
        // Write success is indicated by returning length of the structure.
        if (dwRet != sizeof(DWORD))
        {
            RETAILMSG(1, (TEXT("\r\nERROR HPRF_ReadWriteEEPROM: Write fail.\r\n")));
            goto CleanUp;
        }
    }

    // Done
    bRetVal = TRUE;

CleanUp:
    // Don't forget to close the handle.
    if (hDriver != INVALID_HANDLE_VALUE)
        CloseHandle(hDriver);
    
return bRetVal;
}

//-----------------------------------------------------------------------------
//
//  Function: VerifyUserEepParams
//
//  This function reads the EEPROM section data what is stored
//  in the flash, checks the signature and version. If signature
//  is not the expected, then it does a restore to default. Also,
//  this can be used to update certain EEP values based on the
//  version number.
//
//  NOTE: Currently HPRF driver acts as EEPROM strcture manager.
//  In the future, we can create a stand-alone driver to manage
//  EEPROM and this kind of checking and use this function in that
//  common driver. For now, since no one else is using this block in EEPROM,
//  this is ok.
//  
//  Parameters:
//          None
//
//  Returns:
//      Returns TRUE on success. Returns FALSE on failure.
//
//-----------------------------------------------------------------------------
BOOL VerifyUserEepParams(void)
{
    EEP_PARAMS eep_param;

    // Clear the structure
    memset(&eep_param, 0, sizeof (EEP_PARAMS));

    // Read the data from the offset (we don't check the return restuls, since the driver returns
    // false if the flash is completely blank for the first time.
    HPRF_ReadWriteEEPROM(&eep_param, 1);

    //RETAILMSG(1, (TEXT("VerifyUserEepParams: Found Sign: 0x%x, Ver: %d\r\n"), eep_param.dwSignature, eep_param.dwVersion));

    // First check the signature
    if (eep_param.dwSignature != EEP_SIGNATURE)
    {
        RETAILMSG(1, (TEXT("EEP VerifyUserEepParams: Using defaults.\r\n")));
        // Empty flash. Restore default values if needed.

        // <<-- Add here if you have to restore any defaults.

        // At the end, put the signature and current version.
        eep_param.dwSignature = EEP_SIGNATURE;
        eep_param.dwVersion = EEP_VERSION;
        return FALSE;

    }
    //RETAILMSG(1, (TEXT("VerifyUserEepParams: Done.\r\n")));
    return TRUE;
}
////////////////////////////////////////////////////////////////////
//              End EEPROM managemebt                   ////////////
////////////////////////////////////////////////////////////////////

//------------------------------------------------------------------------------
//
// Function: ReadProductIdAndRevision
//
// This function returns the product ID and Revision ID of the board
// The GPIO ports are already configured correctly in PanelInfo.cpp file
// before we start this driver (order of DLL is 1 for PanelInfo)
//
// Parameters:
//      panelID
//          [out] returns the product iD
//      revisionID
//          [out] returns the revision iD
//
// Returns:
//      None.
//
//------------------------------------------------------------------------------
void  ReadProductIdAndRevision(DWORD* panelID, DWORD* revisionID)
{
    UINT32 panelId = 0, revisionId = 0;

    // Read the product ID
    if(DDKGpioReadData(PRODUCT_ID_GPIO_PORT, 0xFFFFFFFF, &panelId))
        panelId = (panelId & PRODUCT_ID_MASK) >> 7;

    // Read the revision ID
    if(DDKGpioReadData(REVISION_ID_GPIO_PORT, 0xFFFFFFFF, &revisionId))
        revisionId = (revisionId & REVISION_ID_MASK) >> 13;

    *panelID = panelId;
    *revisionID = revisionId;

    return ;
}

#if 0
/**
 *
 * \author      Prasad Madhur
 *
 * \brief       Gets the power table index based on the RF
 *              module and the table present in the file
 *              hprfgw_rfrec.c:
 *              RFTransmitPowerValuesInDbm[][]
 *              RFPowerLmts[][]
 *
 * \detail
 *
 * \date        11/30/2010
 *
 * \param       void
 *
 * \return      UINT8
 *
 * \retval      Returns the index to the power table
 */
UINT8 GetRFPowerTableIndex(void)
{
    //TODO: Reimplement this in a way not dependent on windows.
    UINT8 index = DEFAULT_POWER_LIMIT_TABLE;
    EEP_PARAMS pEepParam;
    BOOL bReadOK;
    HKEY    hKey;
    UINT32  dwDefTable = 0;
    UINT32  dwSize;

    //RETAILMSG(1, (TEXT("GetRFPowerTableIndex\r\n")));
    bReadOK = HPRF_ReadWriteEEPROM(&pEepParam, 1);      // read the ER parameters from the data store

    if(bReadOK)
    {
        // See if this is a valid table
        if((pEepParam.RfPowerTable & EEP_PT_SIG_MASK) == EEP_PT_SIGNATURE)
        {
            // Yes, use power table index
            index = pEepParam.RfPowerTable & ~EEP_PT_SIG_MASK;
        }
        else
        {
            /* No, return the index for the lowest power table */
            // The data store did not have a valid value. We have set it to a default value.
            // Now, we will look at the registry to see if there is a different default that is needed.
            if (RegOpenKeyEx(HKEY_LOCAL_MACHINE, TEXT("Drivers\\BuiltIn\\HPRF"), 0 , 0, &hKey) == ERROR_SUCCESS)
            {
                dwSize = sizeof(dwDefTable);
                if ( RegQueryValueEx(hKey, L"HPRFDefaultPowerTable", 0, NULL,(LPBYTE)&dwDefTable, &dwSize ) == ERROR_SUCCESS )
                {
                    index = dwDefTable;         // got a default table number, use it.
                    //RETAILMSG(1, (TEXT("GetRFPowerTableIndex Registry default table = 0x%X.\r\n"), index));
                }
                else
                {
                    RETAILMSG(1, (TEXT("GetRFPowerTableIndex Registry default table not found.\r\n")));
                }
                // Close registry key
                RegCloseKey(hKey);
            }
            else
                RETAILMSG(1, (TEXT("GetRFPowerTableIndex: failed to open registry key Drivers\\BuiltIn\\HPRF.\r\n")));
        }
    }
    else
    {
        RETAILMSG(1, (TEXT("GetRFPowerTableIndex: failed to open critical store.\r\n")));
    }

    if(index > MAX_POWER_LIMIT_TABLE)
        index = DEFAULT_POWER_LIMIT_TABLE;

    //RETAILMSG(1, (TEXT("GetRFPowerTableIndex: returned index = 0x%x\r\n"),index));
    return (index);
}
#endif
